const mongoose = require('mongoose');

const subscriptionSchema = new mongoose.Schema({
  // Organization this subscription belongs to
  organizationId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Organization',
    required: true
  },
  
  // Subscription Tier Information
  tier: {
    type: String,
    enum: ['basic', 'professional', 'enterprise'],
    required: true,
    default: 'basic'
  },
  
  // Stripe Integration
  stripeCustomerId: {
    type: String,
    required: true
  },
  stripeSubscriptionId: {
    type: String,
    required: true
  },
  stripePriceId: {
    type: String,
    required: true
  },
  
  // Subscription Status
  status: {
    type: String,
    enum: ['active', 'canceled', 'past_due', 'unpaid', 'trialing'],
    default: 'active'
  },
  
  // Billing Information
  currentPeriodStart: {
    type: Date,
    required: true
  },
  currentPeriodEnd: {
    type: Date,
    required: true
  },
  
  // Pricing
  amount: {
    type: Number,
    required: true
  },
  currency: {
    type: String,
    default: 'usd'
  },
  billingCycle: {
    type: String,
    enum: ['monthly', 'yearly'],
    default: 'monthly'
  },
  
  // Feature Limits
  limits: {
    transportsPerMonth: {
      type: Number,
      default: 50 // Basic tier limit
    },
    technicians: {
      type: Number,
      default: 5 // Basic tier limit
    },
    locations: {
      type: Number,
      default: 1 // Basic tier limit
    },
    apiCallsPerMonth: {
      type: Number,
      default: 1000 // Basic tier limit
    },
    storageGB: {
      type: Number,
      default: 10 // Basic tier limit
    }
  },
  
  // Feature Access
  features: {
    // Tier 1: Basic Features
    caseTracking: {
      type: Boolean,
      default: true
    },
    transportCreation: {
      type: Boolean,
      default: true
    },
    qrCodeGeneration: {
      type: Boolean,
      default: true
    },
    mobileAppAccess: {
      type: Boolean,
      default: true
    },
    pdfReports: {
      type: Boolean,
      default: true
    },
    basicEmailSupport: {
      type: Boolean,
      default: true
    },
    
    // Tier 2: Professional Features
    voiceLoggingAssistant: {
      type: Boolean,
      default: false
    },
    familyWebPortal: {
      type: Boolean,
      default: false
    },
    automatedNotifications: {
      type: Boolean,
      default: false
    },
    multiLocationManagement: {
      type: Boolean,
      default: false
    },
    analyticsDashboard: {
      type: Boolean,
      default: false
    },
    prioritySupport: {
      type: Boolean,
      default: false
    },
    
    // Tier 3: Enterprise Features
    apiIntegrations: {
      type: Boolean,
      default: false
    },
    blockchainVerification: {
      type: Boolean,
      default: false
    },
    customBrandedQrCodes: {
      type: Boolean,
      default: false
    },
    customizableForms: {
      type: Boolean,
      default: false
    },
    aiVoiceAgent: {
      type: Boolean,
      default: false
    },
    dedicatedAccountManager: {
      type: Boolean,
      default: false
    },
    complianceReporting: {
      type: Boolean,
      default: false
    }
  },
  
  // Metadata
  createdAt: {
    type: Date,
    default: Date.now
  },
  updatedAt: {
    type: Date,
    default: Date.now
  },
  canceledAt: {
    type: Date,
    default: null
  }
}, {
  timestamps: true
});

// Update features based on tier
subscriptionSchema.pre('save', function(next) {
  const tierFeatures = {
    basic: {
      caseTracking: true,
      qrCodeGeneration: true,
      mobileAppAccess: true,
      pdfReports: true,
      basicEmailSupport: true,
      voiceLoggingAssistant: true,
      familyWebPortal: false,
      automatedNotifications: false,
      multiLocationManagement: false,
      analyticsDashboard: false,
      prioritySupport: false,
      apiIntegrations: false,
      blockchainVerification: false,
      customBrandedQrCodes: false,
      customizableForms: false,
      aiVoiceAgent: true,
      dedicatedAccountManager: false,
      complianceReporting: false
    },
    professional: {
      caseTracking: true,
      qrCodeGeneration: true,
      mobileAppAccess: true,
      pdfReports: true,
      basicEmailSupport: true,
      voiceLoggingAssistant: true,
      familyWebPortal: true,
      automatedNotifications: true,
      multiLocationManagement: true,
      analyticsDashboard: true,
      prioritySupport: true,
      apiIntegrations: false,
      blockchainVerification: false,
      customBrandedQrCodes: false,
      customizableForms: false,
      aiVoiceAgent: false,
      dedicatedAccountManager: false,
      complianceReporting: false
    },
    enterprise: {
      caseTracking: true,
      qrCodeGeneration: true,
      mobileAppAccess: true,
      pdfReports: true,
      basicEmailSupport: true,
      voiceLoggingAssistant: true,
      familyWebPortal: true,
      automatedNotifications: true,
      multiLocationManagement: true,
      analyticsDashboard: true,
      prioritySupport: true,
      apiIntegrations: true,
      blockchainVerification: true,
      customBrandedQrCodes: true,
      customizableForms: true,
      aiVoiceAgent: true,
      dedicatedAccountManager: true,
      complianceReporting: true
    }
  };

  // Update limits based on tier
  const tierLimits = {
    basic: {
      transportsPerMonth: 50,
      technicians: 5,
      locations: 1,
      apiCallsPerMonth: 1000,
      storageGB: 10
    },
    professional: {
      transportsPerMonth: 200,
      technicians: 20,
      locations: 5,
      apiCallsPerMonth: 5000,
      storageGB: 50
    },
    enterprise: {
      transportsPerMonth: -1, // Unlimited
      technicians: -1, // Unlimited
      locations: -1, // Unlimited
      apiCallsPerMonth: -1, // Unlimited
      storageGB: 500
    }
  };

  if (this.isModified('tier') || this.isNew) {
    this.features = tierFeatures[this.tier];
    this.limits = tierLimits[this.tier];
  }

  next();
});

// Static method to get tier pricing
subscriptionSchema.statics.getTierPricing = function() {
  return {
    basic: {
      name: 'Basic / Essential',
      price: 149,
      currency: 'usd',
      billingCycle: 'monthly',
      description: 'Perfect for small funeral homes and single locations',
      features: [
        'Case tracking with digital custody logs',
        'QR code generation & scanning',
        'Mobile technician app access',
        'PDF report export',
        'Basic email support'
      ]
    },
    professional: {
      name: 'Professional / Growth',
      price: 349,
      currency: 'usd',
      billingCycle: 'monthly',
      description: 'Ideal for mid-sized firms with multiple cases weekly',
      features: [
        'Everything in Basic, plus:',
        'Voice logging assistant (AI voice-to-text)',
        'Family web portal',
        'Automated notifications (SMS/email)',
        'Multi-location management',
        'Analytics dashboard',
        'Priority support (email + phone)'
      ]
    },
    enterprise: {
      name: 'Enterprise / White-Glove',
      price: 599,
      currency: 'usd',
      billingCycle: 'monthly',
      description: 'For large firms and corporate networks',
      features: [
        'Everything in Professional, plus:',
        'API + integrations',
        'Blockchain verification',
        'Custom branded QR codes',
        'Customizable custody forms',
        'AI voice agent integration',
        'Dedicated account manager',
        'Compliance reporting'
      ]
    }
  };
};

// Method to check if a feature is available
subscriptionSchema.methods.hasFeature = function(featureName) {
  return this.features[featureName] === true;
};

// Method to check if within limits
subscriptionSchema.methods.checkLimit = function(limitName, currentValue) {
  const limit = this.limits[limitName];
  if (limit === -1) return true; // Unlimited
  return currentValue < limit;
};

module.exports = mongoose.model('Subscription', subscriptionSchema);
