const mongoose = require('mongoose');

const transportSchema = new mongoose.Schema({
  // Basic Transport Information
  caseId: {
    type: String,
    required: true,
    unique: true,
    index: true
  },
  transportType: {
    type: String,
    enum: ['removal', 'transfer', 'cremation', 'burial', 'repatriation'],
    required: true
  },
  status: {
    type: String,
    enum: [
      'pending',
      'scheduled',
      'in_progress',
      'pickup_completed',
      'in_transit',
      'delivery_completed',
      'completed',
      'cancelled',
      'delayed',
      'emergency'
    ],
    default: 'pending'
  },
  priority: {
    type: String,
    enum: ['low', 'normal', 'high', 'urgent'],
    default: 'normal'
  },
  
  // Deceased Information
  deceased: {
    firstName: {
      type: String,
      required: true,
      trim: true
    },
    lastName: {
      type: String,
      required: true,
      trim: true
    },
    dateOfBirth: {
      type: Date,
      required: true
    },
    dateOfDeath: {
      type: Date,
      required: true
    },
    causeOfDeath: String,
    placeOfDeath: {
      type: String,
      required: true
    },
    timeOfDeath: Date,
    medicalExaminerCase: {
      type: Boolean,
      default: false
    },
    infectiousRisk: {
      type: Boolean,
      default: false
    },
    infectiousDetails: String,
    specialConditions: [String], // implants, pacemaker, etc.
    uniqueIdTag: {
      type: String,
      required: true,
      unique: true
    }
  },
  
  // Location Information
  pickupLocation: {
    name: String,
    address: {
      street: String,
      city: String,
      state: String,
      zipCode: String,
      country: {
        type: String,
        default: 'US'
      }
    },
    coordinates: {
      latitude: Number,
      longitude: Number
    },
    contactPerson: {
      name: String,
      phone: String,
      title: String
    },
    facilityType: {
      type: String,
      enum: ['hospital', 'nursing_home', 'private_residence', 'coroner_office', 'other']
    }
  },
  
  deliveryLocation: {
    name: String,
    address: {
      street: String,
      city: String,
      state: String,
      zipCode: String,
      country: {
        type: String,
        default: 'US'
      }
    },
    coordinates: {
      latitude: Number,
      longitude: Number
    },
    contactPerson: {
      name: String,
      phone: String,
      title: String
    },
    facilityType: {
      type: String,
      enum: ['funeral_home', 'crematory', 'cemetery', 'airport', 'other']
    }
  },
  
  // Scheduling Information
  scheduledPickupTime: {
    type: Date,
    required: true
  },
  scheduledDeliveryTime: {
    type: Date,
    required: true
  },
  actualPickupTime: Date,
  actualDeliveryTime: Date,
  estimatedDuration: Number, // in minutes
  
  // Personnel Information
  assignedTechnician: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: true
  },
  backupTechnician: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User'
  },
  funeralHomeStaff: [{
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User'
  }],
  
  // Organization Information
  organizationId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Organization',
    required: true
  },
  clientId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Client'
  },
  
  // Chain of Custody Events
  custodyEvents: [{
    eventType: {
      type: String,
      enum: [
        'pickup_initiated',
        'identity_confirmed',
        'personal_effects_inventoried',
        'shrouding_completed',
        'vehicle_loaded',
        'pickup_completed',
        'transport_started',
        'gps_checkpoint',
        'emergency_stop',
        'delivery_arrived',
        'identity_verified_delivery',
        'delivery_completed',
        'custody_transferred',
        'voice_log',
        'status_updated',
        'photo_documentation',
        'qr_code_scanned',
        'emergency_alert'
      ],
      required: true
    },
    timestamp: {
      type: Date,
      default: Date.now
    },
    location: {
      coordinates: {
        latitude: Number,
        longitude: Number
      },
      address: String
    },
    performedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User',
      required: true
    },
    notes: String,
    photos: [{
      url: String,
      ipfsHash: String,
      metadata: {
        timestamp: Date,
        gps: {
          latitude: Number,
          longitude: Number
        }
      }
    }],
    voiceLog: {
      url: String,
      ipfsHash: String,
      duration: Number
    },
    qrCodeScanned: {
      code: String,
      location: String
    },
    blockchainTransaction: {
      txHash: String,
      blockNumber: Number,
      timestamp: Date
    }
  }],
  
  // GPS Tracking and Status Updates
  currentLocation: {
    coordinates: {
      latitude: Number,
      longitude: Number
    },
    address: String,
    timestamp: Date,
    accuracy: Number,
    speed: Number,
    heading: Number
  },
  
  statusHistory: [{
    status: {
      type: String,
      enum: [
        'pending',
        'scheduled',
        'in_progress',
        'pickup_completed',
        'in_transit',
        'delivery_completed',
        'completed',
        'cancelled',
        'delayed',
        'emergency'
      ]
    },
    updatedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User',
      required: true
    },
    updatedAt: {
      type: Date,
      default: Date.now
    },
    location: {
      coordinates: {
        latitude: Number,
        longitude: Number
      },
      address: String,
      accuracy: Number
    },
    notes: String,
    photos: [{
      photoUrl: String,
      photoId: String,
      timestamp: Date,
      metadata: {
        gps: {
          latitude: Number,
          longitude: Number,
          accuracy: Number
        },
        device: String,
        orientation: String
      }
    }],
    voiceLogs: [{
      _id: {
        type: mongoose.Schema.Types.ObjectId,
        auto: true
      },
      userId: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'User',
        required: true
      },
      message: {
        type: String,
        required: true,
        trim: true
      },
      audioData: {
        type: String, // Base64 encoded audio data
        required: false
      },
      duration: {
        type: Number, // Duration in seconds
        required: false,
        min: 0
      },
      type: {
        type: String,
        enum: ['voice', 'manual', 'ai'],
        default: 'voice',
        required: true
      },
      status: {
        type: String,
        enum: ['pending', 'transcribed', 'processed', 'failed'],
        default: 'pending',
        required: true
      },
      location: {
        coordinates: {
          latitude: Number,
          longitude: Number
        },
        address: String,
        timestamp: Date
      },
      notes: {
        type: String,
        trim: true
      },
      timestamp: {
        type: Date,
        default: Date.now
      }
    }]
  }],

  // Real-time tracking
  isTracking: {
    type: Boolean,
    default: false
  },
  
  trackingStartedAt: Date,
  trackingEndedAt: Date,
  
  // Estimated times based on current location
  estimatedPickupTime: Date,
  estimatedDeliveryTime: Date,
  
  // Route information
  route: {
    distance: Number, // in kilometers
    duration: Number, // in minutes
    waypoints: [{
      coordinates: {
        latitude: Number,
        longitude: Number
      },
      address: String,
      timestamp: Date,
      status: String
    }]
  },

  // Family notifications
  lastNotificationSent: {
    type: Date,
    default: Date.now
  },
  
  notificationPreferences: {
    statusUpdates: {
      type: Boolean,
      default: true
    },
    locationUpdates: {
      type: Boolean,
      default: true
    },
    delayAlerts: {
      type: Boolean,
      default: true
    },
    emergencyAlerts: {
      type: Boolean,
      default: true
    }
  },
  
  // Documents and Permits
  documents: [{
    type: {
      type: String,
      enum: [
        'death_certificate',
        'burial_transit_permit',
        'cremation_authorization',
        'body_release',
        'medical_examiner_release',
        'airline_documents',
        'consular_documents',
        'other'
      ],
      required: true
    },
    name: String,
    url: String,
    ipfsHash: String,
    uploadedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    },
    uploadedAt: {
      type: Date,
      default: Date.now
    },
    isVerified: {
      type: Boolean,
      default: false
    },
    verifiedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    },
    verifiedAt: Date
  }],
  
  // Personal Effects
  personalEffects: [{
    item: String,
    description: String,
    condition: String,
    location: String,
    handedTo: {
      name: String,
      relationship: String,
      signature: String
    },
    timestamp: {
      type: Date,
      default: Date.now
    }
  }],
  
  // Emergency Information
  emergencyAlerts: [{
    type: {
      type: String,
      enum: ['delay', 'accident', 'medical', 'weather', 'other']
    },
    description: String,
    timestamp: {
      type: Date,
      default: Date.now
    },
    reportedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    },
    resolved: {
      type: Boolean,
      default: false
    },
    resolution: String,
    resolvedAt: Date
  }],
  
  // Communication Log
  communications: [{
    type: {
      type: String,
      enum: ['sms', 'email', 'voice', 'in_app']
    },
    direction: {
      type: String,
      enum: ['inbound', 'outbound']
    },
    recipient: String,
    content: String,
    timestamp: {
      type: Date,
      default: Date.now
    },
    status: {
      type: String,
      enum: ['sent', 'delivered', 'read', 'failed']
    }
  }],
  
  // Family Information
  familyMembers: [{
    user: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    },
    relationship: String,
    isPrimaryContact: {
      type: Boolean,
      default: false
    },
    notificationsEnabled: {
      type: Boolean,
      default: true
    }
  }],
  
  // Blockchain Information
  blockchain: {
    smartContractAddress: String,
    custodyEvents: [{
      eventId: String,
      txHash: String,
      blockNumber: Number,
      timestamp: Date,
      eventType: String
    }],
    documents: [{
      documentId: String,
      ipfsHash: String,
      txHash: String,
      timestamp: Date
    }]
  },
  
  // AI Assistant Information
  aiAssistant: {
    voiceLogs: [{
      sessionId: String,
      url: String,
      ipfsHash: String,
      duration: Number,
      transcript: String,
      timestamp: Date
    }],
    taskReminders: [{
      task: String,
      dueTime: Date,
      completed: {
        type: Boolean,
        default: false
      },
      completedAt: Date
    }]
  },
  
  // Compliance and Audit
  compliance: {
    hipaaCompliant: {
      type: Boolean,
      default: true
    },
    gdprCompliant: {
      type: Boolean,
      default: true
    },
    auditTrail: [{
      action: String,
      performedBy: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'User'
      },
      timestamp: {
        type: Date,
        default: Date.now
      },
      details: String,
      ipAddress: String
    }]
  },
  
  // Notes and Comments
  notes: [{
    content: String,
    author: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    },
    timestamp: {
      type: Date,
      default: Date.now
    },
    isPrivate: {
      type: Boolean,
      default: false
    }
  }],
  
  // Timestamps
  createdAt: {
    type: Date,
    default: Date.now
  },
  updatedAt: {
    type: Date,
    default: Date.now
  },
  completedAt: Date,
  cancelledAt: Date,
  cancelledBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User'
  },
  cancellationReason: String
}, {
  timestamps: true,
  toJSON: { virtuals: true },
  toObject: { virtuals: true }
});

// Virtual for full deceased name
transportSchema.virtual('deceased.fullName').get(function() {
  return `${this.deceased.firstName} ${this.deceased.lastName}`;
});

// Virtual for transport duration
transportSchema.virtual('duration').get(function() {
  if (this.actualPickupTime && this.actualDeliveryTime) {
    return this.actualDeliveryTime - this.actualPickupTime;
  }
  return null;
});

// Virtual for is delayed
transportSchema.virtual('isDelayed').get(function() {
  if (this.scheduledDeliveryTime && this.actualDeliveryTime) {
    return this.actualDeliveryTime > this.scheduledDeliveryTime;
  }
  return false;
});

// Indexes (caseId index is already created by index: true)
transportSchema.index({ organizationId: 1, status: 1 });
transportSchema.index({ assignedTechnician: 1, status: 1 });
transportSchema.index({ 'deceased.uniqueIdTag': 1 });
transportSchema.index({ scheduledPickupTime: 1 });
transportSchema.index({ createdAt: -1 });
transportSchema.index({ 'custodyEvents.timestamp': -1 });

// Pre-save middleware to update updatedAt
transportSchema.pre('save', function(next) {
  this.updatedAt = Date.now();
  next();
});

// Static method to find active transports by organization
transportSchema.statics.findActiveByOrg = function(organizationId) {
  return this.find({
    organizationId,
    status: { $in: ['scheduled', 'in_progress', 'pickup_completed', 'in_transit', 'active', 'pending'] }
  }).populate('assignedTechnician', 'firstName lastName phone');
};

// Static method to find transports by technician
transportSchema.statics.findByTechnician = function(technicianId) {
  return this.find({
    $or: [
      { assignedTechnician: technicianId },
      { backupTechnician: technicianId }
    ]
  }).populate('deceased', 'firstName lastName uniqueIdTag');
};

// Static method to find transports by family member
transportSchema.statics.findByFamilyMember = function(userId) {
  return this.find({
    'familyMembers.user': userId
  }).populate('assignedTechnician', 'firstName lastName phone');
};

// Instance method to add custody event
transportSchema.methods.addCustodyEvent = function(eventData) {
  this.custodyEvents.push(eventData);
  return this.save();
};

// Instance method to update status
transportSchema.methods.updateStatus = function(newStatus, userId) {
  this.status = newStatus;
  this.compliance.auditTrail.push({
    action: `Status changed to ${newStatus}`,
    performedBy: userId,
    details: `Transport status updated from ${this.status} to ${newStatus}`
  });
  return this.save();
};

module.exports = mongoose.model('Transport', transportSchema);
