const express = require('express');
const router = express.Router();
const { verifyToken, authorize } = require('../middleware/auth');
const { checkFeatureAccess } = require('../middleware/subscription');
const { rateLimiter } = require('../middleware/rateLimiter');
const { blockchainService } = require('../services/blockchain');
const Transport = require('../models/Transport');
const User = require('../models/User');

// Apply rate limiting to all blockchain routes
router.use(rateLimiter);

/**
 * @route GET /api/blockchain/events
 * @desc Get blockchain events for a transport
 * @access Private
 */
router.get('/events', verifyToken, checkFeatureAccess('blockchainVerification'), async (req, res) => {
  try {
    const { transportId, eventType, startDate, endDate } = req.query;
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const query = { organizationId: user.organizationId };

    if (transportId) {
      query.transportId = transportId;
    }

    if (eventType) {
      query.eventType = eventType;
    }

    if (startDate && endDate) {
      query.timestamp = {
        $gte: new Date(startDate),
        $lte: new Date(endDate)
      };
    }

    const events = await blockchainService.getEvents(query);

    res.json({
      success: true,
      events
    });
  } catch (error) {
    console.error('Error fetching blockchain events:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch blockchain events'
    });
  }
});

/**
 * @route POST /api/blockchain/events
 * @desc Record a new blockchain event
 * @access Private
 */
router.post('/events', verifyToken, checkFeatureAccess('blockchainVerification'), async (req, res) => {
  try {
    const {
      transportId,
      eventType,
      eventData,
      location,
      timestamp,
      metadata
    } = req.body;

    if (!transportId || !eventType || !eventData) {
      return res.status(400).json({
        success: false,
        message: 'Transport ID, event type, and event data are required'
      });
    }

    // Verify transport exists and user has access
    const transport = await Transport.findById(transportId);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }

    const userId = req.user.id;
    const user = await User.findById(userId);

    if (transport.organizationId.toString() !== user.organizationId.toString()) {
      return res.status(403).json({
        success: false,
        message: 'Access denied to this transport'
      });
    }

    const event = await blockchainService.recordEvent({
      transportId,
      eventType,
      eventData,
      location,
      timestamp: timestamp || new Date(),
      metadata: {
        ...metadata,
        recordedBy: userId,
        organizationId: user.organizationId
      }
    });

    res.status(201).json({
      success: true,
      message: 'Blockchain event recorded successfully',
      event
    });
  } catch (error) {
    console.error('Error recording blockchain event:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to record blockchain event'
    });
  }
});

/**
 * @route GET /api/blockchain/verify/:transportId
 * @desc Verify transport chain of custody on blockchain
 * @access Private
 */
router.get('/verify/:transportId', verifyToken, checkFeatureAccess('blockchainVerification'), async (req, res) => {
  try {
    const { transportId } = req.params;
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    // Verify transport exists and user has access
    const transport = await Transport.findById(transportId);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }

    if (transport.organizationId.toString() !== user.organizationId.toString()) {
      return res.status(403).json({
        success: false,
        message: 'Access denied to this transport'
      });
    }

    const verification = await blockchainService.verifyTransport(transportId);

    res.json({
      success: true,
      verification
    });
  } catch (error) {
    console.error('Error verifying transport:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to verify transport'
    });
  }
});

/**
 * @route GET /api/blockchain/stats
 * @desc Get blockchain statistics
 * @access Private (Admin/Staff only)
 */
router.get('/stats', verifyToken, authorize(['admin', 'staff']), async (req, res) => {
  try {
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const stats = await blockchainService.getStats(user.organizationId);

    res.json({
      success: true,
      stats
    });
  } catch (error) {
    console.error('Error fetching blockchain stats:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch blockchain statistics'
    });
  }
});

/**
 * @route GET /api/blockchain/contract/:address
 * @desc Get smart contract information
 * @access Private (Admin only)
 */
router.get('/contract/:address', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const { address } = req.params;
    const contractInfo = await blockchainService.getContractInfo(address);

    res.json({
      success: true,
      contract: contractInfo
    });
  } catch (error) {
    console.error('Error fetching contract info:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch contract information'
    });
  }
});

/**
 * @route POST /api/blockchain/deploy
 * @desc Deploy new smart contract (admin only)
 * @access Private (Admin only)
 */
router.post('/deploy', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const { contractType, parameters } = req.body;
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const deployment = await blockchainService.deployContract({
      contractType,
      parameters,
      organizationId: user.organizationId,
      deployedBy: userId
    });

    res.status(201).json({
      success: true,
      message: 'Smart contract deployed successfully',
      deployment
    });
  } catch (error) {
    console.error('Error deploying contract:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to deploy smart contract'
    });
  }
});

/**
 * @route GET /api/blockchain/transactions/:transportId
 * @desc Get blockchain transactions for a transport
 * @access Private
 */
router.get('/transactions/:transportId', verifyToken, async (req, res) => {
  try {
    const { transportId } = req.params;
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    // Verify transport exists and user has access
    const transport = await Transport.findById(transportId);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }

    if (transport.organizationId.toString() !== user.organizationId.toString()) {
      return res.status(403).json({
        success: false,
        message: 'Access denied to this transport'
      });
    }

    const transactions = await blockchainService.getTransactions(transportId);

    res.json({
      success: true,
      transactions
    });
  } catch (error) {
    console.error('Error fetching transactions:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch blockchain transactions'
    });
  }
});

/**
 * @route POST /api/blockchain/audit
 * @desc Generate blockchain audit report
 * @access Private (Admin only)
 */
router.post('/audit', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const { startDate, endDate, organizationId } = req.body;
    const userId = req.user.id;
    const user = await User.findById(userId);

    const targetOrgId = organizationId || user.organizationId;

    if (!targetOrgId) {
      return res.status(400).json({
        success: false,
        message: 'Organization ID is required'
      });
    }

    const auditReport = await blockchainService.generateAuditReport({
      startDate,
      endDate,
      organizationId: targetOrgId
    });

    res.json({
      success: true,
      auditReport
    });
  } catch (error) {
    console.error('Error generating audit report:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to generate audit report'
    });
  }
});

module.exports = router;
