const express = require('express');
const router = express.Router();
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const { v4: uuidv4 } = require('uuid');
const Document = require('../models/Document');
const { verifyToken, authorize } = require('../middleware/auth');
const { uploadToIPFS, getFromIPFS } = require('../services/ipfs');
const { verifyDocument } = require('../services/blockchain');

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = 'uploads/documents';
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueName = `${uuidv4()}-${Date.now()}${path.extname(file.originalname)}`;
    cb(null, uniqueName);
  }
});

const upload = multer({
  storage: storage,
  limits: {
    fileSize: 10 * 1024 * 1024, // 10MB limit
  },
  fileFilter: (req, file, cb) => {
    console.log('Multer fileFilter - File:', {
      originalname: file.originalname,
      mimetype: file.mimetype,
      size: file.size
    });
    
    const allowedExtensions = /\.(jpeg|jpg|png|gif|pdf|doc|docx)$/i;
    const allowedMimeTypes = /^(image\/(jpeg|jpg|png|gif)|application\/(pdf|msword|vnd\.openxmlformats-officedocument\.wordprocessingml\.document))$/;
    
    const hasValidExtension = allowedExtensions.test(file.originalname);
    const hasValidMimeType = allowedMimeTypes.test(file.mimetype);

    console.log('File validation:', {
      hasValidExtension,
      hasValidMimeType,
      extension: path.extname(file.originalname),
      mimetype: file.mimetype
    });

    if (hasValidExtension && hasValidMimeType) {
      return cb(null, true);
    } else {
      const error = `File type not allowed. Extension: ${path.extname(file.originalname)}, MimeType: ${file.mimetype}`;
      console.log('File rejected:', error);
      cb(new Error(error));
    }
  }
});

// Get all documents for the authenticated user
router.get('/', verifyToken, async (req, res) => {
  try {
    const { role, organizationId } = req.user;
    let query = {};

    if (role === 'family') {
      query = { familyId: req.user.id };
    } else if (role === 'technician') {
      query = { technicianId: req.user.id };
    } else if (role === 'staff' || role === 'admin') {
      query = { organizationId };
    }

    const documents = await Document.find(query)
      .populate('uploadedBy', 'firstName lastName')
      .sort({ uploadDate: -1 });

    res.json({
      success: true,
      documents
    });
  } catch (error) {
    console.error('Error fetching documents:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch documents'
    });
  }
});

// Get documents for family members
router.get('/family', verifyToken, authorize('family'), async (req, res) => {
  try {
    const documents = await Document.find({ familyId: req.user.id })
      .populate('uploadedBy', 'firstName lastName')
      .sort({ uploadDate: -1 });

    res.json({
      success: true,
      documents
    });
  } catch (error) {
    console.error('Error fetching family documents:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch family documents'
    });
  }
});

// Get documents for organization
router.get('/organization', verifyToken, authorize('staff', 'admin', 'funeral_home_staff'), async (req, res) => {
  try {
    const documents = await Document.find({ organizationId: req.user.organizationId })
      .populate('uploadedBy', 'firstName lastName')
      .populate('familyId', 'firstName lastName')
      .populate('technicianId', 'firstName lastName')
      .sort({ createdAt: -1 });

    res.json({
      success: true,
      documents
    });
  } catch (error) {
    console.error('Error fetching organization documents:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch organization documents'
    });
  }
});

// Get documents for technicians
router.get('/technician', verifyToken, authorize('technician'), async (req, res) => {
  try {
    const documents = await Document.find({ 
      technicianId: req.user._id,
      organizationId: req.user.organizationId
    })
      .populate('uploadedBy', 'firstName lastName')
      .populate('transportId', 'caseId deceased.firstName deceased.lastName')
      .sort({ createdAt: -1 });

    res.json({
      success: true,
      documents
    });
  } catch (error) {
    console.error('Error fetching technician documents:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch technician documents'
    });
  }
});

// Get documents by transport ID
router.get('/transport/:transportId', verifyToken, async (req, res) => {
  try {
    const { transportId } = req.params;
    const { role, _id, organizationId } = req.user;

    let query = { transportId, organizationId };

    // Filter by role permissions
    if (role === 'family') {
      query.familyId = _id;
    } else if (role === 'technician') {
      query.technicianId = _id;
    }

    const documents = await Document.find(query)
      .populate('uploadedBy', 'firstName lastName')
      .populate('familyId', 'firstName lastName')
      .populate('technicianId', 'firstName lastName')
      .sort({ createdAt: -1 });

    res.json({
      success: true,
      documents
    });
  } catch (error) {
    console.error('Error fetching transport documents:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch transport documents'
    });
  }
});

// Get documents by type
router.get('/type/:documentType', verifyToken, async (req, res) => {
  try {
    const { documentType } = req.params;
    const { role, _id, organizationId } = req.user;

    let query = { documentType, organizationId };

    // Filter by role permissions
    if (role === 'family') {
      query.familyId = _id;
    } else if (role === 'technician') {
      query.technicianId = _id;
    }

    const documents = await Document.find(query)
      .populate('uploadedBy', 'firstName lastName')
      .populate('familyId', 'firstName lastName')
      .populate('technicianId', 'firstName lastName')
      .populate('transportId', 'caseId deceased.firstName deceased.lastName')
      .sort({ createdAt: -1 });

    res.json({
      success: true,
      documents
    });
  } catch (error) {
    console.error('Error fetching documents by type:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch documents by type'
    });
  }
});

// Upload a new document
router.post('/', verifyToken, (req, res, next) => {
  upload.single('document')(req, res, (err) => {
    if (err instanceof multer.MulterError) {
      console.error('Multer error:', err);
      return res.status(400).json({
        success: false,
        message: `File upload error: ${err.message}`
      });
    } else if (err) {
      console.error('File validation error:', err);
      return res.status(400).json({
        success: false,
        message: `File validation error: ${err.message}`
      });
    }
    next();
  });
}, async (req, res) => {
  try {
    console.log('Document upload request received:', {
      body: req.body,
      file: req.file ? {
        filename: req.file.filename,
        originalname: req.file.originalname,
        mimetype: req.file.mimetype,
        size: req.file.size,
        path: req.file.path
      } : null,
      user: {
        id: req.user._id,
        role: req.user.role,
        organizationId: req.user.organizationId
      }
    });

    const { title, documentType, description, expiryDate, transportId, tags } = req.body;
    const file = req.file;

    if (!file) {
      console.log('No file uploaded');
      return res.status(400).json({
        success: false,
        message: 'No file uploaded'
      });
    }

    // Validate document type based on user role
    const allowedDocumentTypes = getAllowedDocumentTypes(req.user.role);
    if (!allowedDocumentTypes.includes(documentType)) {
      console.log('Invalid document type:', documentType, 'for role:', req.user.role);
      return res.status(400).json({
        success: false,
        message: `Document type '${documentType}' is not allowed for your role`
      });
    }

    console.log('Uploading to IPFS:', file.path, file.filename);
    // Upload to IPFS
    let ipfsResult;
    try {
      ipfsResult = await uploadToIPFS(file.path, file.filename);
      console.log('IPFS upload result:', ipfsResult);
    } catch (ipfsError) {
      console.error('IPFS upload failed:', ipfsError);
      // Clean up local file
      if (fs.existsSync(file.path)) {
        fs.unlinkSync(file.path);
      }
      return res.status(500).json({
        success: false,
        message: 'Failed to upload file to IPFS: ' + ipfsError.message
      });
    }
    
    // Create document record
    const documentData = {
      filename: file.filename,
      originalName: file.originalname,
      mimeType: file.mimetype,
      size: file.size,
      storageType: 'ipfs',
      storagePath: file.path,
      ipfsHash: ipfsResult.hash,
      documentType,
      description,
      tags: tags ? tags.split(',').map(tag => tag.trim()) : [],
      organizationId: req.user.organizationId,
      uploadedBy: req.user._id,
      status: 'pending_approval',
      approvalStatus: 'pending',
      checksum: require('crypto').createHash('sha256').update(file.path).digest('hex')
    };

    // Add role-specific data
    if (req.user.role === 'family') {
      documentData.familyId = req.user._id;
      documentData.allowedRoles = ['family', 'staff', 'admin'];
      if (transportId) {
        // Verify that the family member has access to this transport
        const Transport = require('../models/Transport');
        const transport = await Transport.findOne({
          _id: transportId,
          'familyMembers.user': req.user._id,
          organizationId: req.user.organizationId
        });
        
        if (!transport) {
          return res.status(403).json({
            success: false,
            message: 'Access denied to this transport'
          });
        }
        
        documentData.transportId = transportId;
      }
    } else if (req.user.role === 'technician') {
      documentData.technicianId = req.user._id;
      documentData.allowedRoles = ['technician', 'staff', 'admin'];
      if (transportId) documentData.transportId = transportId;
    } else if (req.user.role === 'staff' || req.user.role === 'admin') {
      documentData.allowedRoles = ['staff', 'admin'];
      if (transportId) documentData.transportId = transportId;
    }

    if (expiryDate) {
      documentData.retentionExpiry = new Date(expiryDate);
    }

    console.log('Creating document record:', documentData);
    
    let document;
    try {
      document = new Document(documentData);
      await document.save();
      console.log('Document saved successfully:', document._id);
    } catch (saveError) {
      console.error('Document save failed:', saveError);
      // Clean up local file
      if (fs.existsSync(file.path)) {
        fs.unlinkSync(file.path);
      }
      return res.status(500).json({
        success: false,
        message: 'Failed to save document: ' + saveError.message
      });
    }

    // Verify document on blockchain
    try {
      await verifyDocument(document._id, ipfsResult.hash);
      document.isVerified = true;
      await document.save();
      console.log('Document verified on blockchain');
    } catch (blockchainError) {
      console.error('Blockchain verification failed:', blockchainError);
    }

    // Clean up local file
    try {
      if (fs.existsSync(file.path)) {
        fs.unlinkSync(file.path);
        console.log('Local file cleaned up');
      }
    } catch (cleanupError) {
      console.error('File cleanup failed:', cleanupError);
    }

    console.log('Document upload completed successfully');
    res.status(201).json({
      success: true,
      message: 'Document uploaded successfully',
      document
    });
  } catch (error) {
    console.error('Error uploading document:', error);
    
    // Clean up local file if it exists
    if (req.file && req.file.path && fs.existsSync(req.file.path)) {
      try {
        fs.unlinkSync(req.file.path);
        console.log('Cleaned up local file after error');
      } catch (cleanupError) {
        console.error('Failed to cleanup file after error:', cleanupError);
      }
    }
    
    res.status(500).json({
      success: false,
      message: 'Failed to upload document: ' + error.message
    });
  }
});

// Helper function to get allowed document types by role
function getAllowedDocumentTypes(role) {
  const documentTypes = {
    family: [
      'death_certificate',
      'medical_examiner_report',
      'family_id_document',
      'family_contact_info',
      'special_instructions',
      'legal_documents',
      'power_of_attorney',
      'executor_documents',
      'family_consent',
      'family_photo'
    ],
    technician: [
      'checkpoint_photo',
      'condition_report',
      'incident_report',
      'completion_certificate',
      'gps_tracking_log',
      'pickup_verification',
      'delivery_verification',
      'custody_verification'
    ],
    staff: [
      'transport_permit',
      'chain_of_custody_form',
      'compliance_certificate',
      'insurance_document',
      'liability_document',
      'route_planning',
      'emergency_contacts',
      'regulatory_compliance',
      'medical_records',
      'release_form',
      'transfer_authorization',
      'facility_documentation'
    ],
    admin: [
      'death_certificate',
      'medical_examiner_report',
      'transport_permit',
      'chain_of_custody_form',
      'compliance_certificate',
      'insurance_document',
      'liability_document',
      'route_planning',
      'emergency_contacts',
      'regulatory_compliance',
      'checkpoint_photo',
      'condition_report',
      'incident_report',
      'completion_certificate',
      'gps_tracking_log',
      'pickup_verification',
      'delivery_verification',
      'medical_records',
      'release_form',
      'transfer_authorization',
      'facility_documentation',
      'family_consent',
      'cremation_authorization',
      'burial_permit',
      'family_photo',
      'custody_verification',
      'other'
    ]
  };
  
  return documentTypes[role] || [];
}

// Get a specific document
router.get('/:id', verifyToken, async (req, res) => {
  try {
    const document = await Document.findById(req.params.id)
      .populate('uploadedBy', 'firstName lastName')
      .populate('familyId', 'firstName lastName');

    if (!document) {
      return res.status(404).json({
        success: false,
        message: 'Document not found'
      });
    }

    // Check access permissions
    const { role, id, organizationId } = req.user;
    const hasAccess = 
      document.uploadedBy._id.toString() === id ||
      document.familyId?._id.toString() === id ||
      document.technicianId?.toString() === id ||
      (role === 'staff' || role === 'admin') && document.organizationId?.toString() === organizationId;

    if (!hasAccess) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }

    res.json({
      success: true,
      document
    });
  } catch (error) {
    console.error('Error fetching document:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch document'
    });
  }
});

// Download document from IPFS
router.get('/:id/download', verifyToken, async (req, res) => {
  try {
    const document = await Document.findById(req.params.id);

    if (!document) {
      return res.status(404).json({
        success: false,
        message: 'Document not found'
      });
    }

    // Check access permissions
    const { role, id, organizationId } = req.user;
    const hasAccess = 
      document.uploadedBy.toString() === id ||
      document.familyId?.toString() === id ||
      document.technicianId?.toString() === id ||
      (role === 'staff' || role === 'admin') && document.organizationId?.toString() === organizationId;

    if (!hasAccess) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }

    // Get file from IPFS
    const fileBuffer = await getFromIPFS(document.ipfsHash);

    res.setHeader('Content-Type', `application/${document.fileType}`);
    res.setHeader('Content-Disposition', `attachment; filename="${document.originalName}"`);
    res.send(fileBuffer);
  } catch (error) {
    console.error('Error downloading document:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to download document'
    });
  }
});

// Update document status
router.put('/:id/status', verifyToken, authorize('staff', 'admin', 'funeral_home_staff'), async (req, res) => {
  try {
    const { status, notes } = req.body;
    const document = await Document.findById(req.params.id);

    if (!document) {
      return res.status(404).json({
        success: false,
        message: 'Document not found'
      });
    }

    // Check organization access
    if (document.organizationId?.toString() !== req.user.organizationId?.toString()) {
      console.log('Organization access check failed:', {
        documentOrgId: document.organizationId?.toString(),
        userOrgId: req.user.organizationId?.toString(),
        documentId: document._id,
        userId: req.user._id
      });
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }

    document.status = status;
    if (notes) document.notes = notes;
    document.reviewedBy = req.user._id;
    document.reviewDate = new Date();

    await document.save();

    res.json({
      success: true,
      message: 'Document status updated',
      document
    });
  } catch (error) {
    console.error('Error updating document status:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update document status'
    });
  }
});

// Approve or reject document
router.put('/:id/approval', verifyToken, authorize('staff', 'admin', 'funeral_home_staff'), async (req, res) => {
  try {
    const { approvalStatus, rejectionReason, revisionNotes } = req.body;
    const document = await Document.findById(req.params.id);

    if (!document) {
      return res.status(404).json({
        success: false,
        message: 'Document not found'
      });
    }

    // Check organization access
    if (document.organizationId?.toString() !== req.user.organizationId?.toString()) {
      console.log('Organization access check failed:', {
        documentOrgId: document.organizationId?.toString(),
        userOrgId: req.user.organizationId?.toString(),
        documentId: document._id,
        userId: req.user._id
      });
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }

    document.approvalStatus = approvalStatus;
    document.approvedBy = req.user._id;
    document.approvedAt = new Date();

    if (approvalStatus === 'rejected' && rejectionReason) {
      document.rejectionReason = rejectionReason;
    }

    if (approvalStatus === 'requires_revision' && revisionNotes) {
      document.revisionNotes = revisionNotes;
      document.requiresRevision = true;
    }

    if (approvalStatus === 'approved') {
      document.status = 'ready';
    }

    await document.save();

    res.json({
      success: true,
      message: `Document ${approvalStatus}`,
      document
    });
  } catch (error) {
    console.error('Error updating document approval:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update document approval'
    });
  }
});

// Verify document
router.put('/:id/verify', verifyToken, authorize('staff', 'admin', 'funeral_home_staff'), async (req, res) => {
  try {
    const { verificationNotes } = req.body;
    const document = await Document.findById(req.params.id);

    if (!document) {
      return res.status(404).json({
        success: false,
        message: 'Document not found'
      });
    }

    // Check organization access
    if (document.organizationId?.toString() !== req.user.organizationId?.toString()) {
      console.log('Organization access check failed:', {
        documentOrgId: document.organizationId?.toString(),
        userOrgId: req.user.organizationId?.toString(),
        documentId: document._id,
        userId: req.user._id
      });
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }

    document.isVerified = true;
    document.verifiedBy = req.user._id;
    document.verifiedAt = new Date();
    if (verificationNotes) {
      document.verificationNotes = verificationNotes;
    }

    await document.save();

    res.json({
      success: true,
      message: 'Document verified',
      document
    });
  } catch (error) {
    console.error('Error verifying document:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to verify document'
    });
  }
});

// Share document with family
router.put('/:id/share-family', verifyToken, authorize('staff', 'admin', 'funeral_home_staff'), async (req, res) => {
  try {
    const { expiryDate } = req.body;
    const document = await Document.findById(req.params.id);

    if (!document) {
      return res.status(404).json({
        success: false,
        message: 'Document not found'
      });
    }

    // Check organization access
    if (document.organizationId?.toString() !== req.user.organizationId?.toString()) {
      console.log('Organization access check failed:', {
        documentOrgId: document.organizationId?.toString(),
        userOrgId: req.user.organizationId?.toString(),
        documentId: document._id,
        userId: req.user._id
      });
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }

    await document.shareWithFamily(req.user._id, expiryDate ? new Date(expiryDate) : null);

    res.json({
      success: true,
      message: 'Document shared with family',
      document
    });
  } catch (error) {
    console.error('Error sharing document with family:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to share document with family'
    });
  }
});

// Revoke family access
router.put('/:id/revoke-family', verifyToken, authorize('staff', 'admin', 'funeral_home_staff'), async (req, res) => {
  try {
    const document = await Document.findById(req.params.id);

    if (!document) {
      return res.status(404).json({
        success: false,
        message: 'Document not found'
      });
    }

    // Check organization access
    if (document.organizationId?.toString() !== req.user.organizationId?.toString()) {
      console.log('Organization access check failed:', {
        documentOrgId: document.organizationId?.toString(),
        userOrgId: req.user.organizationId?.toString(),
        documentId: document._id,
        userId: req.user._id
      });
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }

    await document.revokeFamilyAccess();

    res.json({
      success: true,
      message: 'Family access revoked',
      document
    });
  } catch (error) {
    console.error('Error revoking family access:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to revoke family access'
    });
  }
});

// Delete document
router.delete('/:id', verifyToken, async (req, res) => {
  try {
    const document = await Document.findById(req.params.id);

    if (!document) {
      return res.status(404).json({
        success: false,
        message: 'Document not found'
      });
    }

    // Check permissions
    const { role, id, organizationId } = req.user;
    const canDelete = 
      document.uploadedBy.toString() === id ||
      (role === 'admin' && document.organizationId?.toString() === organizationId);

    if (!canDelete) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }

    await Document.findByIdAndDelete(req.params.id);

    res.json({
      success: true,
      message: 'Document deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting document:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete document'
    });
  }
});

module.exports = router;
