const express = require('express');
const router = express.Router();
const { verifyToken, authorize } = require('../middleware/auth');
const { 
  sendVoiceNotification, 
  sendVoiceCall, 
  sendMultiChannelNotification,
  sendEmergencyNotification,
  sendStatusUpdateNotification,
  sendReminderNotification,
  sendComplianceAlert
} = require('../services/notification');
const { aiService } = require('../services/ai');

// Apply authentication middleware to all routes
router.use(verifyToken);

// Get unread notifications count
router.get('/unread-count', async (req, res) => {
  try {
    const Notification = require('../models/Notification');
    const count = await Notification.getUnreadCount(req.user._id);
    
    res.json({
      success: true,
      count: count
    });
  } catch (error) {
    console.error('Error getting unread count:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to get unread count',
      error: error.message
    });
  }
});

// Send voice notification (replaces SMS)
router.post('/voice', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { phoneNumber, message, context } = req.body;

    if (!phoneNumber || !message) {
      return res.status(400).json({
        success: false,
        message: 'Phone number and message are required'
      });
    }

    const result = await sendVoiceNotification(phoneNumber, message, { context });
    
    res.json({
      success: true,
      data: result,
      message: 'Voice notification sent successfully'
    });
  } catch (error) {
    console.error('Voice notification error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to send voice notification',
      error: error.message
    });
  }
});

// Send AI voice call
router.post('/voice-call', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { phoneNumber, message, context } = req.body;

    if (!phoneNumber || !message) {
      return res.status(400).json({
        success: false,
        message: 'Phone number and message are required'
      });
    }

    const result = await sendVoiceCall(phoneNumber, message, { context });
    
    res.json({
      success: true,
      data: result,
      message: 'AI voice call initiated successfully'
    });
  } catch (error) {
    console.error('Voice call error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to initiate voice call',
      error: error.message
    });
  }
});

// Family AI voice call - allows family members to initiate AI voice calls
router.post('/family-ai-call', authorize('family'), async (req, res) => {
  try {
    const { phoneNumber, transportContext, message } = req.body;
    const { user } = req;

    if (!phoneNumber) {
      return res.status(400).json({
        success: false,
        message: 'Phone number is required'
      });
    }

    // Create context-aware message for family AI call
    const familyCallMessage = message || `Hello ${user.firstName || 'Family Member'}, this is your AI assistant from Obitix. I'm here to help you with any questions about your transport, provide updates, or offer support during this difficult time. ${transportContext ? `I can see you have transport information available: ${transportContext}` : ''} How can I assist you today?`;

    // Create AI voice call with family context
    const result = await aiService.createRetellCall(
      phoneNumber, 
      null, 
      familyCallMessage, 
      'family-support'
    );
    
    res.json({
      success: true,
      data: {
        callId: result.callId,
        status: result.status,
        message: 'AI voice call initiated successfully',
        callType: 'family-support'
      },
      message: 'AI voice call initiated successfully'
    });
  } catch (error) {
    console.error('Family AI voice call error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to initiate AI voice call',
      error: error.message
    });
  }
});

// Get Retell call status
router.get('/call-status/:callId', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { callId } = req.params;
    
    const result = await aiService.getRetellCallStatus(callId);
    
    res.json({
      success: true,
      data: result
    });
  } catch (error) {
    console.error('Call status check error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to get call status',
      error: error.message
    });
  }
});

// Send emergency notification
router.post('/emergency', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { transportId, emergencyType, details } = req.body;

    if (!transportId || !emergencyType || !details) {
      return res.status(400).json({
        success: false,
        message: 'Transport ID, emergency type, and details are required'
      });
    }

    // Get transport data (you'll need to import Transport model)
    const Transport = require('../models/Transport');
    const transport = await Transport.findById(transportId)
      .populate('assignedTechnician')
      .populate('backupTechnician')
      .populate('familyMembers.user');

    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }

    const result = await sendEmergencyNotification(transport, emergencyType, details);
    
    res.json({
      success: true,
      data: result,
      message: 'Emergency notifications sent successfully'
    });
  } catch (error) {
    console.error('Emergency notification error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to send emergency notifications',
      error: error.message
    });
  }
});

// Send status update notification
router.post('/status-update', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { transportId, status, notes } = req.body;

    if (!transportId || !status) {
      return res.status(400).json({
        success: false,
        message: 'Transport ID and status are required'
      });
    }

    // Get transport data
    const Transport = require('../models/Transport');
    const transport = await Transport.findById(transportId)
      .populate('familyMembers.user');

    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }

    const result = await sendStatusUpdateNotification(transport, status, notes);
    
    res.json({
      success: true,
      data: result,
      message: 'Status update notifications sent successfully'
    });
  } catch (error) {
    console.error('Status update notification error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to send status update notifications',
      error: error.message
    });
  }
});

// Send reminder notification
router.post('/reminder', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { userId, reminder } = req.body;

    if (!userId || !reminder) {
      return res.status(400).json({
        success: false,
        message: 'User ID and reminder data are required'
      });
    }

    // Get user data
    const User = require('../models/User');
    const user = await User.findById(userId);

    if (!user) {
      return res.status(404).json({
        success: false,
        message: 'User not found'
      });
    }

    const result = await sendReminderNotification(user, reminder);
    
    res.json({
      success: true,
      data: result,
      message: 'Reminder notification sent successfully'
    });
  } catch (error) {
    console.error('Reminder notification error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to send reminder notification',
      error: error.message
    });
  }
});

// Send compliance alert
router.post('/compliance-alert', authorize('admin'), async (req, res) => {
  try {
    const { userId, alert } = req.body;

    if (!userId || !alert) {
      return res.status(400).json({
        success: false,
        message: 'User ID and alert data are required'
      });
    }

    // Get user data
    const User = require('../models/User');
    const user = await User.findById(userId);

    if (!user) {
      return res.status(404).json({
        success: false,
        message: 'User not found'
      });
    }

    const result = await sendComplianceAlert(user, alert);
    
    res.json({
      success: true,
      data: result,
      message: 'Compliance alert sent successfully'
    });
  } catch (error) {
    console.error('Compliance alert error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to send compliance alert',
      error: error.message
    });
  }
});

// Test voice notification with message enhancement
router.post('/test-voice', authorize('admin'), async (req, res) => {
  try {
    const { phoneNumber, message, context } = req.body;

    if (!phoneNumber || !message) {
      return res.status(400).json({
        success: false,
        message: 'Phone number and message are required'
      });
    }

    // Test message enhancement
    const enhancedMessage = await aiService.enhanceNotificationMessage(message, context || {});
    
    // Send voice notification with enhanced message
    const result = await sendVoiceNotification(phoneNumber, enhancedMessage, { context });
    
    res.json({
      success: true,
      data: {
        originalMessage: message,
        enhancedMessage: enhancedMessage,
        callResult: result
      },
      message: 'Test voice notification sent successfully'
    });
  } catch (error) {
    console.error('Test voice notification error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to send test voice notification',
      error: error.message
    });
  }
});

module.exports = router;
