const express = require('express');
const router = express.Router();
const { authorize } = require('../middleware/auth');
const Transport = require('../models/Transport');

// Get all voice logs for the authenticated user
router.get('/', authorize('technician', 'admin'), async (req, res) => {
  try {
    const userId = req.user._id;
    const organizationId = req.user.organizationId;

    // Get all transports for the user's organization
    const transports = await Transport.find({
      organizationId: organizationId,
      $or: [
        { assignedTechnician: userId },
        { backupTechnician: userId }
      ]
    }).populate('assignedTechnician', 'firstName lastName email');

    // Extract all voice logs from transports
    let allVoiceLogs = [];
    transports.forEach(transport => {
      if (transport.custodyEvents && transport.custodyEvents.length > 0) {
        transport.custodyEvents.forEach(event => {
          if (event.voiceLogs && event.voiceLogs.length > 0) {
            event.voiceLogs.forEach(log => {
              // Only include logs created by the current user
              if (log.userId.toString() === userId.toString()) {
                allVoiceLogs.push({
                  ...log.toObject(),
                  transportId: transport._id,
                  transportCaseId: transport.caseId,
                  transportDeceased: transport.deceased
                });
              }
            });
          }
        });
      }
    });

    // Sort by timestamp (newest first)
    allVoiceLogs.sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp));

    res.json({
      success: true,
      data: {
        voiceLogs: allVoiceLogs,
        total: allVoiceLogs.length
      }
    });
  } catch (error) {
    console.error('Error fetching voice logs:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch voice logs',
      error: error.message
    });
  }
});

// Get voice logs for a specific transport
router.get('/transport/:transportId', authorize('technician', 'admin', 'family'), async (req, res) => {
  try {
    const { transportId } = req.params;
    const userId = req.user._id;
    const organizationId = req.user.organizationId;
    const userRole = req.user.role;

    // Get the transport
    const transport = await Transport.findById(transportId);
    
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }

    // Check if user has access to this transport
    const hasAccess = 
      transport.organizationId.toString() === organizationId.toString() &&
      (userRole === 'family' || 
       transport.assignedTechnician.toString() === userId.toString() ||
       transport.backupTechnician?.toString() === userId.toString() ||
       userRole === 'admin');

    if (!hasAccess) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to access this transport'
      });
    }

    // Extract voice logs from all custody events
    let transportLogs = [];
    if (transport.custodyEvents && transport.custodyEvents.length > 0) {
      transport.custodyEvents.forEach(event => {
        if (event.voiceLogs && event.voiceLogs.length > 0) {
          event.voiceLogs.forEach(log => {
            // Family members can see all logs, others see their own
            if (userRole === 'family' || log.userId.toString() === userId.toString()) {
              transportLogs.push({
                ...log.toObject(),
                transportId: transport._id,
                transportCaseId: transport.caseId,
                transportDeceased: transport.deceased
              });
            }
          });
        }
      });
    }

    // Sort by timestamp (newest first)
    transportLogs.sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp));

    res.json({
      success: true,
      data: {
        voiceLogs: transportLogs,
        total: transportLogs.length
      }
    });
  } catch (error) {
    console.error('Error fetching transport voice logs:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch transport voice logs',
      error: error.message
    });
  }
});

// Create a new voice log
router.post('/', authorize('technician', 'admin'), async (req, res) => {
  try {
    const { transportId, message, duration = 0, type = 'manual' } = req.body;
    const userId = req.user._id;
    const organizationId = req.user.organizationId;

    if (!transportId || !message) {
      return res.status(400).json({
        success: false,
        message: 'Transport ID and message are required'
      });
    }

    // Validate transport exists and user has access
    const transport = await Transport.findById(transportId);
    
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }

    // Check if user is authorized for this transport
    const isAuthorized = 
      transport.organizationId.toString() === organizationId.toString() &&
      (transport.assignedTechnician.toString() === userId.toString() ||
       transport.backupTechnician?.toString() === userId.toString() ||
       req.user.role === 'admin');

    if (!isAuthorized) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to create voice logs for this transport'
      });
    }

    // Create new voice log
    const newLog = {
      userId: userId,
      message: message,
      duration: duration || 0,
      type: type || 'manual',
      status: type === 'voice' ? 'pending' : 'processed',
      timestamp: new Date()
    };

    // Add voice log to the transport's custody events
    // If no custody events exist, create one
    if (!transport.custodyEvents || transport.custodyEvents.length === 0) {
      transport.custodyEvents = [{
        eventType: 'voice_log',
        timestamp: new Date(),
        userId: userId,
        notes: 'Voice log created',
        voiceLogs: [newLog]
      }];
    } else {
      // Add to the latest custody event or create a new one
      const latestEvent = transport.custodyEvents[transport.custodyEvents.length - 1];
      if (latestEvent.eventType === 'voice_log') {
        latestEvent.voiceLogs.push(newLog);
      } else {
        transport.custodyEvents.push({
          eventType: 'voice_log',
          timestamp: new Date(),
          userId: userId,
          notes: 'Voice log created',
          voiceLogs: [newLog]
        });
      }
    }

    // Save the transport
    await transport.save();

    console.log('Voice log created:', {
      transportId,
      type,
      message: message.substring(0, 50) + '...',
      userId,
      organizationId
    });

    res.status(201).json({
      success: true,
      data: {
        voiceLog: newLog
      },
      message: 'Voice log created successfully'
    });
  } catch (error) {
    console.error('Error creating voice log:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create voice log',
      error: error.message
    });
  }
});

// Record voice audio for transport
router.post('/record', authorize('technician', 'admin'), async (req, res) => {
  try {
    const { transportId, voiceData, notes, duration, latitude, longitude, address } = req.body;
    const userId = req.user._id;
    const organizationId = req.user.organizationId;

    if (!transportId || !voiceData) {
      return res.status(400).json({
        success: false,
        message: 'Transport ID and voice data are required'
      });
    }

    // Validate transport exists and user has access
    const transport = await Transport.findById(transportId);
    
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }

    // Check if user is authorized for this transport
    const isAuthorized = 
      transport.organizationId.toString() === organizationId.toString() &&
      (transport.assignedTechnician.toString() === userId.toString() ||
       transport.backupTechnician?.toString() === userId.toString() ||
       req.user.role === 'admin');

    if (!isAuthorized) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to record voice for this transport'
      });
    }

    // Create voice log entry
    const newLog = {
      userId: userId,
      message: notes || 'Voice recording',
      duration: duration || 0,
      type: 'voice',
      status: 'pending',
      audioData: voiceData, // Store the base64 audio data
      location: {
        coordinates: {
          latitude: latitude || null,
          longitude: longitude || null
        },
        address: address || null,
        timestamp: new Date()
      },
      timestamp: new Date()
    };

    // Add voice log to the transport's custody events
    // If no custody events exist, create one
    if (!transport.custodyEvents || transport.custodyEvents.length === 0) {
      transport.custodyEvents = [{
        eventType: 'voice_log',
        timestamp: new Date(),
        userId: userId,
        notes: 'Voice recording created',
        voiceLogs: [newLog]
      }];
    } else {
      // Add to the latest custody event or create a new one
      const latestEvent = transport.custodyEvents[transport.custodyEvents.length - 1];
      if (latestEvent.eventType === 'voice_log') {
        latestEvent.voiceLogs.push(newLog);
      } else {
        transport.custodyEvents.push({
          eventType: 'voice_log',
          timestamp: new Date(),
          userId: userId,
          notes: 'Voice recording created',
          voiceLogs: [newLog]
        });
      }
    }

    // Save the transport
    await transport.save();

    console.log('Voice recording saved:', {
      transportId,
      duration,
      userId,
      organizationId
    });

    res.status(201).json({
      success: true,
      data: {
        voiceLog: newLog
      },
      message: 'Voice recording saved successfully'
    });
  } catch (error) {
    console.error('Error saving voice recording:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to save voice recording',
      error: error.message
    });
  }
});

// Get voice logs for AI voice agent (family access)
router.get('/ai/family/:transportId', authorize('family'), async (req, res) => {
  try {
    const { transportId } = req.params;
    const userId = req.user._id;
    const organizationId = req.user.organizationId;

    // Get the transport
    const transport = await Transport.findById(transportId);
    
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }

    // Check if user has access to this transport
    const hasAccess = transport.familyMembers.some(member => 
      member.user.toString() === userId.toString()
    );

    if (!hasAccess) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to access this transport'
      });
    }

    // Extract voice logs from all custody events
    let transportLogs = [];
    if (transport.custodyEvents && transport.custodyEvents.length > 0) {
      transport.custodyEvents.forEach(event => {
        if (event.voiceLogs && event.voiceLogs.length > 0) {
          event.voiceLogs.forEach(log => {
            transportLogs.push({
              id: log._id,
              type: log.type,
              message: log.message,
              timestamp: log.timestamp,
              duration: log.duration,
              location: log.location,
              hasAudio: log.type === 'voice' && log.audioData
            });
          });
        }
      });
    }

    // Sort by timestamp (newest first)
    transportLogs.sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp));

    res.json({
      success: true,
      data: {
        voiceLogs: transportLogs,
        total: transportLogs.length,
        transportInfo: {
          caseId: transport.caseId,
          status: transport.status,
          deceasedName: `${transport.deceased.firstName} ${transport.deceased.lastName}`,
          pickupLocation: transport.pickupLocation,
          deliveryLocation: transport.deliveryLocation
        }
      }
    });
  } catch (error) {
    console.error('Error fetching voice logs for AI:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch voice logs for AI',
      error: error.message
    });
  }
});

module.exports = router;
