const stripe = require('stripe')(process.env.STRIPE_SECRET_KEY);
const axios = require('axios');
const Subscription = require('../models/Subscription');
const Organization = require('../models/Organization');

class StripeService {
  constructor() {
    this.prices = {
      basic: process.env.STRIPE_BASIC_PRICE_ID,
      professional: process.env.STRIPE_PROFESSIONAL_PRICE_ID,
      enterprise: process.env.STRIPE_ENTERPRISE_PRICE_ID
    };
  }

  // Create a new customer
  async createCustomer(organization) {
    try {
      const customer = await stripe.customers.create({
        email: organization.contact?.email || organization.email,
        name: organization.name,
        metadata: {
          organizationId: organization._id.toString()
        }
      });

      return customer;
    } catch (error) {
      console.error('Error creating Stripe customer:', error);
      throw new Error('Failed to create customer');
    }
  }

  // Create a subscription
  async createSubscription(organizationId, tier, paymentMethodId = null) {
    try {
      const organization = await Organization.findById(organizationId);
      if (!organization) {
        throw new Error('Organization not found');
      }

      // Get or create Stripe customer
      let customer;
      if (organization.stripeCustomerId) {
        customer = await stripe.customers.retrieve(organization.stripeCustomerId);
      } else {
        customer = await this.createCustomer(organization);
        organization.stripeCustomerId = customer.id;
        await organization.save();
      }

      // Set up payment method if provided
      if (paymentMethodId) {
        await stripe.paymentMethods.attach(paymentMethodId, {
          customer: customer.id,
        });
        await stripe.customers.update(customer.id, {
          invoice_settings: {
            default_payment_method: paymentMethodId,
          },
        });
      }

      // Create subscription
      const subscription = await stripe.subscriptions.create({
        customer: customer.id,
        items: [{ price: this.prices[tier] }],
        payment_behavior: 'default_incomplete',
        payment_settings: { save_default_payment_method: 'on_subscription' },
        expand: ['latest_invoice.payment_intent'],
        metadata: {
          organizationId: organizationId.toString(),
          tier: tier
        }
      });

      // Create subscription record in database
      const subscriptionRecord = new Subscription({
        organizationId: organizationId,
        tier: tier,
        stripeCustomerId: customer.id,
        stripeSubscriptionId: subscription.id,
        stripePriceId: this.prices[tier],
        status: subscription.status,
        currentPeriodStart: new Date(subscription.current_period_start * 1000),
        currentPeriodEnd: new Date(subscription.current_period_end * 1000),
        amount: subscription.items.data[0].price.unit_amount / 100,
        currency: subscription.currency
      });

      await subscriptionRecord.save();

      return {
        subscription: subscriptionRecord,
        stripeSubscription: subscription
      };
    } catch (error) {
      console.error('Error creating subscription:', error);
      throw new Error('Failed to create subscription');
    }
  }

  // Update subscription tier
  async updateSubscription(subscriptionId, newTier) {
    try {
      const subscription = await Subscription.findById(subscriptionId);
      if (!subscription) {
        throw new Error('Subscription not found');
      }

      // Update in Stripe
      const stripeSubscription = await stripe.subscriptions.retrieve(subscription.stripeSubscriptionId);
      const updatedStripeSubscription = await stripe.subscriptions.update(subscription.stripeSubscriptionId, {
        items: [{
          id: stripeSubscription.items.data[0].id,
          price: this.prices[newTier],
        }],
        proration_behavior: 'create_prorations',
        metadata: {
          tier: newTier
        }
      });

      // Update in database
      subscription.tier = newTier;
      subscription.stripePriceId = this.prices[newTier];
      subscription.amount = updatedStripeSubscription.items.data[0].price.unit_amount / 100;
      await subscription.save();

      return subscription;
    } catch (error) {
      console.error('Error updating subscription:', error);
      throw new Error('Failed to update subscription');
    }
  }

  // Cancel subscription
  async cancelSubscription(subscriptionId, cancelAtPeriodEnd = true) {
    try {
      const subscription = await Subscription.findById(subscriptionId);
      if (!subscription) {
        throw new Error('Subscription not found');
      }

      // Cancel in Stripe
      const stripeSubscription = await stripe.subscriptions.update(subscription.stripeSubscriptionId, {
        cancel_at_period_end: cancelAtPeriodEnd
      });

      // Update in database
      subscription.status = cancelAtPeriodEnd ? 'active' : 'canceled';
      if (!cancelAtPeriodEnd) {
        subscription.canceledAt = new Date();
      }
      await subscription.save();

      return subscription;
    } catch (error) {
      console.error('Error canceling subscription:', error);
      throw new Error('Failed to cancel subscription');
    }
  }

  // Reactivate subscription
  async reactivateSubscription(subscriptionId) {
    try {
      const subscription = await Subscription.findById(subscriptionId);
      if (!subscription) {
        throw new Error('Subscription not found');
      }

      // Reactivate in Stripe
      const stripeSubscription = await stripe.subscriptions.update(subscription.stripeSubscriptionId, {
        cancel_at_period_end: false
      });

      // Update in database
      subscription.status = 'active';
      subscription.canceledAt = null;
      await subscription.save();

      return subscription;
    } catch (error) {
      console.error('Error reactivating subscription:', error);
      throw new Error('Failed to reactivate subscription');
    }
  }

  // Get subscription details
  async getSubscription(subscriptionId) {
    try {
      const subscription = await Subscription.findById(subscriptionId)
        .populate('organizationId', 'name email');

      if (!subscription) {
        throw new Error('Subscription not found');
      }

      // Get Stripe subscription details
      const stripeSubscription = await stripe.subscriptions.retrieve(subscription.stripeSubscriptionId);

      return {
        ...subscription.toObject(),
        stripeSubscription
      };
    } catch (error) {
      console.error('Error getting subscription:', error);
      throw new Error('Failed to get subscription');
    }
  }

  // Get organization subscription
  async getOrganizationSubscription(organizationId) {
    try {
      const subscription = await Subscription.findOne({ organizationId })
        .populate('organizationId', 'name email');

      if (!subscription) {
        return null;
      }

      // Try to get Stripe subscription details, but don't fail if it doesn't exist
      let stripeSubscription = null;
      try {
        stripeSubscription = await stripe.subscriptions.retrieve(subscription.stripeSubscriptionId);
      } catch (stripeError) {
        console.warn('Stripe subscription not found, using local data only:', subscription.stripeSubscriptionId);
        // Continue with local subscription data only
      }

      const subscriptionData = subscription.toObject();
      
      // Convert amount from cents to dollars for display
      const formattedAmount = (subscriptionData.amount / 100).toFixed(2);
      
      return {
        ...subscriptionData,
        amount: subscriptionData.amount, // Keep original amount in cents
        formattedAmount: formattedAmount, // Add formatted amount in dollars
        stripeSubscription
      };
    } catch (error) {
      console.error('Error getting organization subscription:', error);
      throw new Error('Failed to get organization subscription');
    }
  }

  // Create checkout session
  async createCheckoutSession(organizationId, tier, successUrl, cancelUrl) {
    try {
      const organization = await Organization.findById(organizationId);
      if (!organization) {
        throw new Error('Organization not found');
      }

      // Get or create Stripe customer
      let customer;
      if (organization.stripeCustomerId) {
        customer = await stripe.customers.retrieve(organization.stripeCustomerId);
      } else {
        customer = await this.createCustomer(organization);
        organization.stripeCustomerId = customer.id;
        await organization.save();
      }

      // Create checkout session
      const session = await stripe.checkout.sessions.create({
        customer: customer.id,
        payment_method_types: ['card'],
        line_items: [{
          price: this.prices[tier],
          quantity: 1,
        }],
        mode: 'subscription',
        success_url: successUrl,
        cancel_url: cancelUrl,
        customer_update: {
          address: 'auto'
        },
        metadata: {
          organizationId: organizationId.toString(),
          tier: tier
        }
      });

      return session;
    } catch (error) {
      console.error('Error creating checkout session:', error);
      throw new Error('Failed to create checkout session');
    }
  }

  // Handle webhook events
  async handleWebhook(event) {
    console.log('Received webhook event:', event.type, event.id);
    
    try {
      switch (event.type) {
        case 'customer.subscription.created':
          console.log('Processing subscription created event');
          await this.handleSubscriptionCreated(event.data.object);
          break;
        case 'customer.subscription.updated':
          console.log('Processing subscription updated event');
          await this.handleSubscriptionUpdated(event.data.object);
          break;
        case 'customer.subscription.deleted':
          console.log('Processing subscription deleted event');
          await this.handleSubscriptionDeleted(event.data.object);
          break;
        case 'invoice.payment_succeeded':
          console.log('Processing payment succeeded event');
          await this.handlePaymentSucceeded(event.data.object);
          break;
        case 'invoice.payment_failed':
          console.log('Processing payment failed event');
          await this.handlePaymentFailed(event.data.object);
          break;
        default:
          console.log(`Unhandled event type: ${event.type}`);
      }
      console.log('Webhook event processed successfully:', event.type);
    } catch (error) {
      console.error('Error handling webhook:', error);
      throw error;
    }
  }

  // Handle subscription created
  async handleSubscriptionCreated(stripeSubscription) {
    console.log('Handling subscription created:', stripeSubscription.id);
    
    // Check if subscription already exists
    let subscription = await Subscription.findOne({
      stripeSubscriptionId: stripeSubscription.id
    });

    if (subscription) {
      // Update existing subscription
      subscription.status = stripeSubscription.status;
      subscription.currentPeriodStart = new Date(stripeSubscription.current_period_start * 1000);
      subscription.currentPeriodEnd = new Date(stripeSubscription.current_period_end * 1000);
      await subscription.save();
      console.log('Updated existing subscription:', subscription._id);
    } else {
      // Create new subscription from checkout session
      const organizationId = stripeSubscription.metadata?.organizationId;
      const tier = stripeSubscription.metadata?.tier;
      
      if (!organizationId || !tier) {
        console.error('Missing organizationId or tier in subscription metadata:', stripeSubscription.metadata);
        return;
      }
      
      // Get the price details
      const priceId = this.prices[tier];
      const price = await stripe.prices.retrieve(priceId);
      
      subscription = new Subscription({
        organizationId: organizationId,
        tier: tier,
        stripeCustomerId: stripeSubscription.customer,
        stripeSubscriptionId: stripeSubscription.id,
        stripePriceId: priceId,
        status: stripeSubscription.status,
        currentPeriodStart: new Date(stripeSubscription.current_period_start * 1000),
        currentPeriodEnd: new Date(stripeSubscription.current_period_end * 1000),
        amount: price.unit_amount / 100,
        currency: price.currency
      });
      
      await subscription.save();
      console.log('Created new subscription:', subscription._id);
    }
  }

  // Handle subscription updated
  async handleSubscriptionUpdated(stripeSubscription) {
    const subscription = await Subscription.findOne({
      stripeSubscriptionId: stripeSubscription.id
    });

    if (subscription) {
      subscription.status = stripeSubscription.status;
      subscription.currentPeriodStart = new Date(stripeSubscription.current_period_start * 1000);
      subscription.currentPeriodEnd = new Date(stripeSubscription.current_period_end * 1000);
      await subscription.save();
    }
  }

  // Handle subscription deleted
  async handleSubscriptionDeleted(stripeSubscription) {
    const subscription = await Subscription.findOne({
      stripeSubscriptionId: stripeSubscription.id
    });

    if (subscription) {
      subscription.status = 'canceled';
      subscription.canceledAt = new Date();
      await subscription.save();
    }
  }

  // Handle payment succeeded
  async handlePaymentSucceeded(invoice) {
    console.log('Handling payment succeeded for subscription:', invoice.subscription);
    
    const subscription = await Subscription.findOne({
      stripeSubscriptionId: invoice.subscription
    });

    if (subscription) {
      subscription.status = 'active';
      await subscription.save();
      console.log('Updated subscription status to active:', subscription._id);
    } else {
      console.warn('Payment succeeded but subscription not found in database:', invoice.subscription);
    }
  }

  // Handle payment failed
  async handlePaymentFailed(invoice) {
    const subscription = await Subscription.findOne({
      stripeSubscriptionId: invoice.subscription
    });

    if (subscription) {
      subscription.status = 'past_due';
      await subscription.save();
    }
  }

  // Get pricing information
  getPricing() {
    return Subscription.getTierPricing();
  }
}

module.exports = new StripeService();
