const axios = require('axios');
require('dotenv').config();

// Test configuration
const API_BASE_URL = process.env.API_BASE_URL || 'https://obitix.prohirely.ltd/api';
const TEST_PHONE = process.env.TEST_PHONE || '+1234567890'; // Replace with your test phone number
const TEST_TOKEN = process.env.TEST_TOKEN || 'your-test-jwt-token'; // Replace with a valid JWT token

// Test functions
async function testVoiceNotification() {
  console.log('🧪 Testing Voice Notification (Replaces SMS)...');
  
  try {
    const response = await axios.post(`${API_BASE_URL}/notifications/voice`, {
      phoneNumber: TEST_PHONE,
      message: 'This is a test notification from Obitix. Your transport status has been updated.',
      context: {
        transportId: 'TEST-001',
        status: 'In Transit',
        location: 'Downtown Medical Center'
      }
    }, {
      headers: {
        'Authorization': `Bearer ${TEST_TOKEN}`,
        'Content-Type': 'application/json'
      }
    });

    console.log('✅ Voice notification sent successfully!');
    console.log('📞 Call ID:', response.data.data.callId);
    console.log('📊 Status:', response.data.data.status);
    console.log('🎯 Type:', response.data.data.type);
    
    return response.data.data.callId;
  } catch (error) {
    console.error('❌ Voice notification failed:', error.response?.data || error.message);
    return null;
  }
}

async function testVoiceCall() {
  console.log('\n🧪 Testing AI Voice Call...');
  
  try {
    const response = await axios.post(`${API_BASE_URL}/notifications/voice-call`, {
      phoneNumber: TEST_PHONE,
      message: 'Hello, this is an AI voice call from Obitix. We have important information about your transport.',
      context: {
        callType: 'information',
        priority: 'high',
        userRole: 'family'
      }
    }, {
      headers: {
        'Authorization': `Bearer ${TEST_TOKEN}`,
        'Content-Type': 'application/json'
      }
    });

    console.log('✅ AI voice call initiated successfully!');
    console.log('📞 Call ID:', response.data.data.callId);
    console.log('📊 Status:', response.data.data.status);
    console.log('🎯 Type:', response.data.data.type);
    
    return response.data.data.callId;
  } catch (error) {
    console.error('❌ AI voice call failed:', error.response?.data || error.message);
    return null;
  }
}

async function testCallStatus(callId) {
  if (!callId) {
    console.log('\n⏭️  Skipping call status check (no call ID available)');
    return;
  }

  console.log(`\n🧪 Checking Call Status for ${callId}...`);
  
  try {
    const response = await axios.get(`${API_BASE_URL}/notifications/call-status/${callId}`, {
      headers: {
        'Authorization': `Bearer ${TEST_TOKEN}`
      }
    });

    console.log('✅ Call status retrieved successfully!');
    console.log('📊 Call Data:', JSON.stringify(response.data.data, null, 2));
  } catch (error) {
    console.error('❌ Call status check failed:', error.response?.data || error.message);
  }
}

async function testMessageEnhancement() {

  
  try {
    const response = await axios.post(`${API_BASE_URL}/notifications/test-voice`, {
      phoneNumber: TEST_PHONE,
      message: 'Transport update: Vehicle arrived at pickup location.',
      context: {
        transportId: 'TEST-002',
        eventType: 'pickup',
        location: 'Memorial Hospital'
      }
    }, {
      headers: {
        'Authorization': `Bearer ${TEST_TOKEN}`,
        'Content-Type': 'application/json'
      }
    });

    console.log('✅ Message enhancement test completed!');
    console.log('📝 Original Message:', response.data.data.originalMessage);
    console.log('✨ Enhanced Message:', response.data.data.enhancedMessage);
    console.log('📞 Call Result:', response.data.data.callResult);
  } catch (error) {
    console.error('❌ Message enhancement test failed:', error.response?.data || error.message);
  }
}

async function testEmergencyNotification() {
  console.log('\n🧪 Testing Emergency Notification...');
  
  try {
    const response = await axios.post(`${API_BASE_URL}/notifications/emergency`, {
      transportId: '507f1f77bcf86cd799439011', // Replace with actual transport ID
      emergencyType: 'Vehicle Breakdown',
      details: 'Transport vehicle has experienced a mechanical issue. Backup vehicle dispatched.'
    }, {
      headers: {
        'Authorization': `Bearer ${TEST_TOKEN}`,
        'Content-Type': 'application/json'
      }
    });

    console.log('✅ Emergency notification sent successfully!');
    console.log('📊 Result:', response.data.data);
  } catch (error) {
    console.error('❌ Emergency notification failed:', error.response?.data || error.message);
  }
}

// Main test runner
async function runTests() {
  console.log('🚀 Starting Retell + OpenAI Voice Notification Tests\n');
  console.log('📋 Test Configuration:');
  console.log(`   API Base URL: ${API_BASE_URL}`);
  console.log(`   Test Phone: ${TEST_PHONE}`);
  console.log(`   Environment: ${process.env.NODE_ENV || 'development'}\n`);

  // Run tests
  const voiceCallId = await testVoiceNotification();
  const aiCallId = await testVoiceCall();
  
  // Wait a moment for calls to be processed
  await new Promise(resolve => setTimeout(resolve, 2000));
  
  await testCallStatus(voiceCallId);
  await testCallStatus(aiCallId);
  await testMessageEnhancement();
  await testEmergencyNotification();

  console.log('\n🎉 All tests completed!');
  console.log('\n📚 What was tested:');
  console.log('   ✅ Voice notifications (replacing SMS)');
  console.log('   ✅ AI voice calls with Retell');
  console.log('   ✅ Call status checking');
  console.log('   ✅ Message enhancement with OpenAI');
  console.log('   ✅ Emergency notifications');
  console.log('\n🔧 Key Benefits:');
  console.log('   🎯 No more Twilio dependency');
  console.log('   🤖 Intelligent AI voice interactions');
  console.log('   💬 Natural, compassionate messaging');
  console.log('   📞 Real-time voice communication');
  console.log('   🔄 Enhanced message context');
}

// Run tests if this file is executed directly
if (require.main === module) {
  runTests().catch(console.error);
}

module.exports = {
  testVoiceNotification,
  testVoiceCall,
  testCallStatus,
  testMessageEnhancement,
  testEmergencyNotification,
  runTests
};
