const OpenAI = require('openai');
require('dotenv').config();

// Initialize OpenAI
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

async function testVoiceGeneration() {
  try {
    console.log('Testing OpenAI TTS service...');
    console.log('API Key configured:', !!process.env.OPENAI_API_KEY);
    
    // Test 1: Check if we can list models
    console.log('\n1. Testing model access...');
    const models = await openai.models.list();
    console.log('Models available:', models.data.length);
    
    // Test 2: Generate speech
    console.log('\n2. Testing speech generation...');
    const speech = await openai.audio.speech.create({
      model: 'tts-1',
      voice: 'alloy',
      input: 'Hello! This is a test voice response from the AI assistant.'
    });
    
    console.log('Speech generated successfully!');
    console.log('Speech buffer size:', speech.arrayBuffer ? 'Available' : 'Not available');
    
    // Test 3: Convert to base64
    console.log('\n3. Testing base64 conversion...');
    const buffer = Buffer.from(await speech.arrayBuffer());
    const base64Audio = buffer.toString('base64');
    
    console.log('Base64 conversion successful!');
    console.log('Audio data length:', base64Audio.length);
    console.log('Audio data preview:', base64Audio.substring(0, 50) + '...');
    
    // Test 4: Test transcription
    console.log('\n4. Testing transcription...');
    // Note: This would require an actual audio file, so we'll skip for now
    
    console.log('\n✅ All voice generation tests passed!');
    
  } catch (error) {
    console.error('\n❌ Voice generation test failed:');
    console.error('Error:', error.message);
    console.error('Error code:', error.code);
    console.error('Error type:', error.type);
    
    if (error.code === 'invalid_api_key') {
      console.error('\n🔑 API Key Issue: Check your OPENAI_API_KEY in .env file');
    } else if (error.code === 'insufficient_quota') {
      console.error('\n💰 Quota Issue: OpenAI account has insufficient credits');
    } else if (error.code === 'rate_limit_exceeded') {
      console.error('\n⏱️ Rate Limit: Too many requests, try again later');
    }
  }
}

testVoiceGeneration();

