const express = require('express');
const router = express.Router();
const { verifyToken, authorize } = require('../middleware/auth');
const { rateLimiter } = require('../middleware/rateLimiter');
const Transport = require('../models/Transport');
const User = require('../models/User');
const Organization = require('../models/Organization');

// Apply rate limiting to all admin routes
router.use(rateLimiter);

// Test route to check if admin routes are working
router.get('/test', verifyToken, (req, res) => {
  console.log('Admin test route - User:', req.user);
  res.json({
    success: true,
    message: 'Admin test route working',
    user: req.user
  });
});

// Test route with authorize middleware
router.get('/test-auth', verifyToken, authorize(['admin']), (req, res) => {
  console.log('Admin test auth route - User:', req.user);
  res.json({
    success: true,
    message: 'Admin test auth route working',
    user: req.user
  });
});

// Test route to check current user role
router.get('/test-role', verifyToken, (req, res) => {
  console.log('Role test route - User:', {
    _id: req.user._id,
    role: req.user.role,
    email: req.user.email
  });
  res.json({
    success: true,
    message: 'Role test route working',
    user: {
      _id: req.user._id,
      role: req.user.role,
      email: req.user.email
    }
  });
});

/**
 * @route GET /api/admin/dashboard/stats
 * @desc Get admin dashboard statistics
 * @access Private (Admin/Staff only)
 */
router.get('/dashboard/stats', verifyToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const user = await User.findById(userId);
    
    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    // Calculate date ranges
    const now = new Date();
    const startOfDay = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const startOfWeek = new Date(now.getFullYear(), now.getMonth(), now.getDate() - now.getDay());

    // Get transport statistics
    const transportStats = await Transport.aggregate([
      { $match: { organizationId: user.organizationId } },
      {
        $group: {
          _id: '$status',
          count: { $sum: 1 }
        }
      }
    ]);

    const totalTransports = transportStats.reduce((sum, stat) => sum + stat.count, 0);
    const activeTransports = transportStats.find(s => s._id === 'active')?.count || 0;
    const completedToday = await Transport.countDocuments({
      organizationId: user.organizationId,
      status: 'completed',
      completedAt: { $gte: startOfDay }
    });
    const completedThisWeek = await Transport.countDocuments({
      organizationId: user.organizationId,
      status: 'completed',
      completedAt: { $gte: startOfWeek }
    });

    // Get staff statistics
    const staffStats = await User.aggregate([
      { $match: { organizationId: user.organizationId } },
      {
        $group: {
          _id: '$role',
          count: { $sum: 1 },
          active: {
            $sum: { $cond: [{ $eq: ['$status', 'active'] }, 1, 0] }
          }
        }
      }
    ]);

    const totalStaff = staffStats.reduce((sum, stat) => sum + stat.count, 0);
    const activeStaff = staffStats.reduce((sum, stat) => sum + stat.active, 0);

    // Calculate average transport time
    const completedTransports = await Transport.find({
      organizationId: user.organizationId,
      status: 'completed',
      completedAt: { $exists: true },
      createdAt: { $exists: true }
    });

    const averageTransportTime = completedTransports.length > 0
      ? completedTransports.reduce((sum, transport) => {
          const duration = new Date(transport.completedAt) - new Date(transport.createdAt);
          return sum + duration;
        }, 0) / completedTransports.length / (1000 * 60) // Convert to minutes
      : 0;

    // Mock customer satisfaction (in real app, this would come from surveys)
    const customerSatisfaction = 4.8; // Out of 5

    res.json({
      success: true,
      stats: {
        totalTransports,
        activeTransports,
        completedToday,
        completedThisWeek,
        totalStaff,
        activeStaff,
        averageTransportTime: Math.round(averageTransportTime),
        customerSatisfaction
      }
    });
  } catch (error) {
    console.error('Error fetching admin stats:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch dashboard statistics'
    });
  }
});

/**
 * @route GET /api/admin/dashboard/activity
 * @desc Get recent activity for admin dashboard
 * @access Private (Admin/Staff only)
 */
router.get('/dashboard/activity', verifyToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const user = await User.findById(userId);
    
    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    // Get recent transports
    const recentTransports = await Transport.find({
      organizationId: user.organizationId
    })
    .sort({ updatedAt: -1 })
    .limit(10)
    .populate('assignedTechnician', 'firstName lastName');

    // Get recent user activities
    const recentUsers = await User.find({
      organizationId: user.organizationId
    })
    .sort({ lastLogin: -1 })
    .limit(5);

    // Combine and format activities
    const activities = [];

    // Add transport activities
    recentTransports.forEach(transport => {
      activities.push({
        _id: transport._id,
        type: 'transport',
        description: `Transport ${transport.caseId} status updated to ${transport.status}`,
        timestamp: transport.updatedAt,
        user: transport.assignedTechnician ? 
          `${transport.assignedTechnician.firstName} ${transport.assignedTechnician.lastName}` : 
          'System'
      });
    });

    // Add user activities
    recentUsers.forEach(user => {
      if (user.lastLogin) {
        activities.push({
          _id: user._id,
          type: 'staff',
          description: `${user.firstName} ${user.lastName} logged in`,
          timestamp: user.lastLogin,
          user: `${user.firstName} ${user.lastName}`
        });
      }
    });

    // Sort by timestamp and limit
    activities.sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp));
    activities.splice(10);

    res.json({
      success: true,
      activities
    });
  } catch (error) {
    console.error('Error fetching admin activity:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch recent activity'
    });
  }
});

/**
 * @route GET /api/admin/transports
 * @desc Get all transports for admin view
 * @access Private (Admin/Staff only)
 */
router.get('/transports', verifyToken, async (req, res) => {
  try {
    const { page = 1, limit = 20, status, search } = req.query;
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const query = { organizationId: user.organizationId };

    if (status) {
      query.status = status;
    }

    if (search) {
      query.$or = [
        { transportId: { $regex: search, $options: 'i' } },
        { 'deceasedInfo.name': { $regex: search, $options: 'i' } },
        { pickupLocation: { $regex: search, $options: 'i' } },
        { destinationLocation: { $regex: search, $options: 'i' } }
      ];
    }

    const transports = await Transport.find(query)
      .populate('technicianId', 'firstName lastName email')
      .populate('familyMembers.userId', 'firstName lastName email')
      .sort({ createdAt: -1 })
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .exec();

    const total = await Transport.countDocuments(query);

    res.json({
      success: true,
      transports,
      pagination: {
        currentPage: parseInt(page),
        totalPages: Math.ceil(total / limit),
        totalTransports: total,
        hasNextPage: page * limit < total,
        hasPrevPage: page > 1
      }
    });
  } catch (error) {
    console.error('Error fetching admin transports:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch transports'
    });
  }
});

/**
 * @route GET /api/admin/staff
 * @desc Get all staff members for admin view (excluding family members)
 * @access Private (Admin only)
 */
router.get('/staff', verifyToken, async (req, res) => {
  try {
    const { page = 1, limit = 20, role, status } = req.query;
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const query = { 
      organizationId: user.organizationId,
      role: { $ne: 'family' } // Exclude family members from staff management
    };

    if (role && role !== 'family') {
      query.role = role;
    }

    if (status) {
      query.status = status;
    }

    const staff = await User.find(query)
      .select('-password')
      .sort({ createdAt: -1 })
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .exec();

    const total = await User.countDocuments(query);

    res.json({
      success: true,
      staff,
      pagination: {
        currentPage: parseInt(page),
        totalPages: Math.ceil(total / limit),
        totalStaff: total,
        hasNextPage: page * limit < total,
        hasPrevPage: page > 1
      }
    });
  } catch (error) {
    console.error('Error fetching admin staff:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch staff members'
    });
  }
});

/**
 * @route POST /api/admin/staff
 * @desc Add new staff member
 * @access Private (Admin only)
 */
router.post('/staff', verifyToken, authorize(['admin']), async (req, res) => {
  console.log('Admin staff creation - User:', {
    _id: req.user._id,
    role: req.user.role,
    email: req.user.email
  });
  console.log('Admin staff creation - Request body:', req.body);
  try {
    const {
      firstName,
      lastName,
      email,
      phone,
      password,
      role,
      permissions,
      licenseNumber
    } = req.body;

    if (!firstName || !lastName || !email || !password || !role) {
      return res.status(400).json({
        success: false,
        message: 'First name, last name, email, password, and role are required'
      });
    }

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return res.status(400).json({
        success: false,
        message: 'Please enter a valid email address'
      });
    }

    // Validate password strength
    if (password.length < 8) {
      return res.status(400).json({
        success: false,
        message: 'Password must be at least 8 characters long'
      });
    }

    // Validate role - allow all user types except admin (admin can only be created during signup)
    const validRoles = ['technician', 'funeral_home_staff', 'family', 'auditor'];
    if (!validRoles.includes(role)) {
      return res.status(400).json({
        success: false,
        message: 'Invalid role specified. Admin accounts can only be created during signup.'
      });
    }

    // Check if user already exists
    const existingUser = await User.findOne({ email });
    if (existingUser) {
      return res.status(400).json({
        success: false,
        message: 'User with this email already exists'
      });
    }

    const userId = req.user._id;
    const adminUser = await User.findById(userId);
    
    // Create new staff member
    const newStaff = new User({
      firstName,
      lastName,
      email,
      phone: phone || '+1234567890', // Provide valid default phone if missing
      password, // Use the password provided by admin
      role,
      permissions: permissions || [],
      organizationId: adminUser.organizationId,
      status: 'active',
      dateOfBirth: role === 'family' ? new Date() : undefined, // Required for family members
      ...(licenseNumber && { 'technicianInfo.licenseNumber': licenseNumber })
    });

    await newStaff.save();

    // Add to organization's staff list
    await Organization.findByIdAndUpdate(adminUser.organizationId, {
      $addToSet: { staff: newStaff._id }
    });

    res.status(201).json({
      success: true,
      message: 'Staff member added successfully',
      staff: {
        _id: newStaff._id,
        firstName: newStaff.firstName,
        lastName: newStaff.lastName,
        email: newStaff.email,
        role: newStaff.role,
        status: newStaff.status
      }
    });
  } catch (error) {
    console.error('Error adding staff member:', error);
    console.error('Error details:', {
      message: error.message,
      stack: error.stack,
      name: error.name
    });

    // Handle validation errors specifically
    if (error.name === 'ValidationError') {
      const validationErrors = {};
      for (const field in error.errors) {
        validationErrors[field] = error.errors[field].message;
      }
      
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: validationErrors
      });
    }

    res.status(500).json({
      success: false,
      message: 'Failed to add staff member',
      error: process.env.NODE_ENV === 'development' ? error.message : undefined
    });
  }
});

/**
 * @route PUT /api/admin/staff/:id
 * @desc Update staff member
 * @access Private (Admin only)
 */
router.put('/staff/:id', verifyToken, async (req, res) => {
  try {
    const { id } = req.params;
    const updateData = req.body;

    // Remove sensitive fields
    delete updateData.password;
    delete updateData._id;

    const staff = await User.findByIdAndUpdate(
      id,
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-password');

    if (!staff) {
      return res.status(404).json({
        success: false,
        message: 'Staff member not found'
      });
    }

    res.json({
      success: true,
      message: 'Staff member updated successfully',
      staff
    });
  } catch (error) {
    console.error('Error updating staff member:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update staff member'
    });
  }
});

/**
 * @route DELETE /api/admin/staff/:id
 * @desc Deactivate staff member
 * @access Private (Admin only)
 */
router.delete('/staff/:id', verifyToken, async (req, res) => {
  try {
    const { id } = req.params;

    const staff = await User.findByIdAndUpdate(
      id,
      { 
        $set: { 
          status: 'inactive',
          deactivatedAt: new Date()
        }
      },
      { new: true }
    );

    if (!staff) {
      return res.status(404).json({
        success: false,
        message: 'Staff member not found'
      });
    }

    res.json({
      success: true,
      message: 'Staff member deactivated successfully'
    });
  } catch (error) {
    console.error('Error deactivating staff member:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to deactivate staff member'
    });
  }
});

/**
 * @route GET /api/admin/reports/transports
 * @desc Get transport reports
 * @access Private (Admin/Staff only)
 */
router.get('/reports/transports', verifyToken, async (req, res) => {
  try {
    const { startDate, endDate, groupBy = 'day' } = req.query;
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const query = { organizationId: user.organizationId };

    if (startDate && endDate) {
      query.createdAt = {
        $gte: new Date(startDate),
        $lte: new Date(endDate)
      };
    }

    let groupFormat;
    switch (groupBy) {
      case 'hour':
        groupFormat = { $dateToString: { format: '%Y-%m-%d %H:00', date: '$createdAt' } };
        break;
      case 'day':
        groupFormat = { $dateToString: { format: '%Y-%m-%d', date: '$createdAt' } };
        break;
      case 'week':
        groupFormat = { $dateToString: { format: '%Y-%U', date: '$createdAt' } };
        break;
      case 'month':
        groupFormat = { $dateToString: { format: '%Y-%m', date: '$createdAt' } };
        break;
      default:
        groupFormat = { $dateToString: { format: '%Y-%m-%d', date: '$createdAt' } };
    }

    const transportStats = await Transport.aggregate([
      { $match: query },
      {
        $group: {
          _id: {
            date: groupFormat,
            status: '$status'
          },
          count: { $sum: 1 }
        }
      },
      {
        $group: {
          _id: '$_id.date',
          statuses: {
            $push: {
              status: '$_id.status',
              count: '$count'
            }
          },
          total: { $sum: '$count' }
        }
      },
      { $sort: { '_id': 1 } }
    ]);

    res.json({
      success: true,
      report: {
        groupBy,
        startDate,
        endDate,
        data: transportStats
      }
    });
  } catch (error) {
    console.error('Error generating transport report:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to generate transport report'
    });
  }
});

/**
 * @route GET /api/admin/reports/staff
 * @desc Get staff performance reports
 * @access Private (Admin only)
 */
router.get('/reports/staff', verifyToken, async (req, res) => {
  try {
    const { startDate, endDate } = req.query;
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const query = { organizationId: user.organizationId };

    if (startDate && endDate) {
      query.createdAt = {
        $gte: new Date(startDate),
        $lte: new Date(endDate)
      };
    }

    const staffPerformance = await Transport.aggregate([
      { $match: query },
      {
        $group: {
          _id: '$technicianId',
          totalTransports: { $sum: 1 },
          completedTransports: {
            $sum: { $cond: [{ $eq: ['$status', 'completed'] }, 1, 0] }
          },
          averageTime: {
            $avg: {
              $cond: [
                { $eq: ['$status', 'completed'] },
                { $subtract: ['$completedAt', '$createdAt'] },
                null
              ]
            }
          }
        }
      },
      {
        $lookup: {
          from: 'users',
          localField: '_id',
          foreignField: '_id',
          as: 'technician'
        }
      },
      {
        $unwind: '$technician'
      },
      {
        $project: {
          technicianName: { $concat: ['$technician.firstName', ' ', '$technician.lastName'] },
          totalTransports: 1,
          completedTransports: 1,
          completionRate: {
            $multiply: [
              { $divide: ['$completedTransports', '$totalTransports'] },
              100
            ]
          },
          averageTimeMinutes: {
            $divide: ['$averageTime', 1000 * 60]
          }
        }
      }
    ]);

    res.json({
      success: true,
      report: {
        startDate,
        endDate,
        data: staffPerformance
      }
    });
  } catch (error) {
    console.error('Error generating staff report:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to generate staff report'
    });
  }
});

/**
 * @route GET /api/admin/settings
 * @desc Get organization settings
 * @access Private (Admin only)
 */
router.get('/settings', verifyToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const organization = await Organization.findById(user.organizationId);

    if (!organization) {
      return res.status(404).json({
        success: false,
        message: 'Organization not found'
      });
    }

    res.json({
      success: true,
      organization: {
        _id: organization._id,
        name: organization.name,
        settings: organization.settings
      }
    });
  } catch (error) {
    console.error('Error fetching organization settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch organization settings'
    });
  }
});

/**
 * @route PUT /api/admin/settings
 * @desc Update organization settings
 * @access Private (Admin only)
 */
router.put('/settings', verifyToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const user = await User.findById(userId);

    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const { name, notifications, features, branding, compliance } = req.body;

    const updateData = {};
    if (name) updateData.name = name;
    if (notifications) updateData['settings.notifications'] = notifications;
    if (features) updateData['settings.features'] = features;
    if (branding) updateData['settings.branding'] = branding;
    if (compliance) updateData['settings.compliance'] = compliance;

    const organization = await Organization.findByIdAndUpdate(
      user.organizationId,
      { $set: updateData },
      { new: true, runValidators: true }
    );

    if (!organization) {
      return res.status(404).json({
        success: false,
        message: 'Organization not found'
      });
    }

    res.json({
      success: true,
      message: 'Organization settings updated successfully',
      organization: {
        _id: organization._id,
        name: organization.name,
        settings: organization.settings
      }
    });
  } catch (error) {
    console.error('Error updating organization settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update organization settings'
    });
  }
});

/**
 * @route GET /api/admin/family-members
 * @desc Get all family members for the organization
 * @access Private (Admin only)
 */
router.get('/family-members', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const userId = req.user._id;
    const user = await User.findById(userId);
    
    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const familyMembers = await User.find({
      organizationId: user.organizationId,
      role: 'family'
    })
    .select('-password')
    .sort({ createdAt: -1 })
    .exec();

    res.json({
      success: true,
      familyMembers
    });
  } catch (error) {
    console.error('Error fetching family members:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch family members'
    });
  }
});

/**
 * @route GET /api/admin/technicians
 * @desc Get all technicians for the organization
 * @access Private (Admin only)
 */
router.get('/technicians', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const userId = req.user._id;
    const user = await User.findById(userId);
    
    if (!user.organizationId) {
      return res.status(400).json({
        success: false,
        message: 'User not associated with any organization'
      });
    }

    const technicians = await User.find({
      organizationId: user.organizationId,
      role: 'technician'
    })
    .select('-password')
    .sort({ createdAt: -1 })
    .exec();

    res.json({
      success: true,
      technicians
    });
  } catch (error) {
    console.error('Error fetching technicians:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch technicians'
    });
  }
});

module.exports = router;
