const express = require('express');
const router = express.Router();
const { verifyToken, authorize } = require('../middleware/auth');
const transportStatusService = require('../services/transportStatus');

/**
 * Update transport status
 * POST /api/transport-status/:transportId/update
 */
router.post('/:transportId/update', verifyToken, authorize(['technician', 'admin', 'staff']), async (req, res) => {
  try {
    const { transportId } = req.params;
    const { status, location, notes, photos, voiceLogs, estimatedPickupTime, estimatedDeliveryTime, allowManualOverride } = req.body;
    const userId = req.user.id;

    const result = await transportStatusService.updateTransportStatus(transportId, userId, {
      status,
      location,
      notes,
      photos,
      voiceLogs,
      estimatedPickupTime,
      estimatedDeliveryTime,
      allowManualOverride
    });

    res.json({
      success: true,
      data: result
    });
  } catch (error) {
    console.error('Error updating transport status:', error);
    res.status(400).json({
      success: false,
      message: error.message
    });
  }
});

/**
 * Update GPS location for active transport
 * POST /api/transport-status/:transportId/gps
 */
router.post('/:transportId/gps', verifyToken, authorize(['technician']), async (req, res) => {
  try {
    const { transportId } = req.params;
    const { coordinates, address, accuracy, speed, heading } = req.body;
    const userId = req.user.id;

    const result = await transportStatusService.updateGPSLocation(transportId, {
      coordinates,
      address,
      accuracy,
      speed,
      heading
    });

    res.json({
      success: true,
      data: result
    });
  } catch (error) {
    console.error('Error updating GPS location:', error);
    res.status(400).json({
      success: false,
      message: error.message
    });
  }
});

/**
 * Start GPS tracking for a transport
 * POST /api/transport-status/:transportId/start-tracking
 */
router.post('/:transportId/start-tracking', verifyToken, authorize(['technician', 'admin']), async (req, res) => {
  try {
    const { transportId } = req.params;
    const userId = req.user.id;

    const result = await transportStatusService.startTracking(transportId, userId);

    res.json({
      success: true,
      data: result
    });
  } catch (error) {
    console.error('Error starting GPS tracking:', error);
    res.status(400).json({
      success: false,
      message: error.message
    });
  }
});

/**
 * Stop GPS tracking for a transport
 * POST /api/transport-status/:transportId/stop-tracking
 */
router.post('/:transportId/stop-tracking', verifyToken, authorize(['technician', 'admin']), async (req, res) => {
  try {
    const { transportId } = req.params;
    const userId = req.user.id;

    const result = await transportStatusService.stopTracking(transportId, userId);

    res.json({
      success: true,
      data: result
    });
  } catch (error) {
    console.error('Error stopping GPS tracking:', error);
    res.status(400).json({
      success: false,
      message: error.message
    });
  }
});

/**
 * Get transport status with live location
 * GET /api/transport-status/:transportId
 */
router.get('/:transportId', verifyToken, async (req, res) => {
  try {
    const { transportId } = req.params;
    const userId = req.user.id;

    const result = await transportStatusService.getTransportStatus(transportId, userId);

    res.json({
      success: true,
      data: result
    });
  } catch (error) {
    console.error('Error getting transport status:', error);
    res.status(400).json({
      success: false,
      message: error.message
    });
  }
});

/**
 * Get all transports for current user with live tracking
 * GET /api/transport-status/user/transports
 */
router.get('/user/transports', verifyToken, async (req, res) => {
  try {
    const userId = req.user.id;
    const userRole = req.user.role;

    const result = await transportStatusService.getUserTransports(userId, userRole);

    res.json({
      success: true,
      data: result
    });
  } catch (error) {
    console.error('Error getting user transports:', error);
    res.status(400).json({
      success: false,
      message: error.message
    });
  }
});

/**
 * Get transport status history
 * GET /api/transport-status/:transportId/history
 */
router.get('/:transportId/history', verifyToken, async (req, res) => {
  try {
    const { transportId } = req.params;
    const userId = req.user.id;

    const result = await transportStatusService.getTransportStatus(transportId, userId);
    
    res.json({
      success: true,
      data: {
        statusHistory: result.transport.statusHistory
      }
    });
  } catch (error) {
    console.error('Error getting transport history:', error);
    res.status(400).json({
      success: false,
      message: error.message
    });
  }
});

/**
 * Get live tracking data for family members
 * GET /api/transport-status/:transportId/live-tracking
 */
router.get('/:transportId/live-tracking', verifyToken, authorize(['family', 'technician', 'admin']), async (req, res) => {
  try {
    const { transportId } = req.params;
    const userId = req.user.id;
    const userRole = req.user.role;

    const result = await transportStatusService.getTransportStatus(transportId, userId);
    
    // Only return essential tracking data for family members
    const trackingData = {
      isTracking: result.transport.isTracking,
      currentLocation: result.transport.currentLocation,
      estimatedPickupTime: result.transport.estimatedPickupTime,
      estimatedDeliveryTime: result.transport.estimatedDeliveryTime,
      status: result.transport.status,
      route: {
        distance: result.transport.route.distance,
        duration: result.transport.route.duration
      }
    };

    res.json({
      success: true,
      data: trackingData
    });
  } catch (error) {
    console.error('Error getting live tracking data:', error);
    res.status(400).json({
      success: false,
      message: error.message
    });
  }
});

module.exports = router;
