const express = require('express');
const router = express.Router();
const Transport = require('../models/Transport');
const { verifyToken, authorize, requirePermission, verifyResourceOwnership } = require('../middleware/auth');
const { checkFeatureAccess } = require('../middleware/subscription');
const { blockchainService } = require('../services/blockchain');
const { aiService } = require('../services/ai');
const qrCodeService = require('../services/qrCode');
const QRCode = require('qrcode');
const { v4: uuidv4 } = require('uuid');

// Apply authentication middleware to all routes
router.use(verifyToken);

// Get all transports for user's organization
router.get('/', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { page = 1, limit = 10, status, transportType, dateFrom, dateTo } = req.query;
    
    const query = { organizationId: req.user.organizationId };
    
    if (status) query.status = status;
    if (transportType) query.transportType = transportType;
    if (dateFrom || dateTo) {
      query.createdAt = {};
      if (dateFrom) query.createdAt.$gte = new Date(dateFrom);
      if (dateTo) query.createdAt.$lte = new Date(dateTo);
    }
    
    const transports = await Transport.find(query)
      .populate('assignedTechnician', 'firstName lastName phone')
      .populate('deceased', 'firstName lastName uniqueIdTag')
      .sort({ createdAt: -1 })
      .limit(limit * 1)
      .skip((page - 1) * limit)
      .exec();
    
    const total = await Transport.countDocuments(query);
    
    res.json({
      success: true,
      data: transports,
      pagination: {
        current: page,
        pages: Math.ceil(total / limit),
        total
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to fetch transports',
      error: error.message
    });
  }
});

// Get transports for technician
router.get('/my-transports', authorize('technician'), async (req, res) => {
  try {
    const { status } = req.query;
    
    const query = {
      $or: [
        { assignedTechnician: req.user._id },
        { backupTechnician: req.user._id }
      ]
    };
    
    if (status) query.status = status;
    
    const transports = await Transport.find(query)
      .populate('deceased', 'firstName lastName uniqueIdTag')
      .populate('pickupLocation', 'name address coordinates')
      .populate('deliveryLocation', 'name address coordinates')
      .sort({ scheduledPickupTime: 1 })
      .exec();
    
    res.json({
      success: true,
      data: transports
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to fetch technician transports',
      error: error.message
    });
  }
});

// Get transports for technician (alias for my-transports)
router.get('/technician', authorize('technician'), async (req, res) => {
  try {
    const { status } = req.query;
    
    const query = {
      $or: [
        { assignedTechnician: req.user._id },
        { backupTechnician: req.user._id }
      ]
    };
    
    if (status) query.status = status;
    
    const transports = await Transport.find(query)
      .populate('deceased', 'firstName lastName uniqueIdTag')
      .populate('pickupLocation', 'name address coordinates')
      .populate('deliveryLocation', 'name address coordinates')
      .sort({ scheduledPickupTime: 1 })
      .exec();
    
    res.json({
      success: true,
      data: transports
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to fetch technician transports',
      error: error.message
    });
  }
});

// Get technician stats
router.get('/technician/stats', authorize('technician'), async (req, res) => {
  try {
    const userId = req.user._id;
    
    // Get total transports assigned to this technician
    const totalTransports = await Transport.countDocuments({
      $or: [
        { assignedTechnician: userId },
        { backupTechnician: userId }
      ]
    });
    
    // Get active transports
    const activeTransports = await Transport.countDocuments({
      $or: [
        { assignedTechnician: userId },
        { backupTechnician: userId }
      ],
      status: { $in: ['scheduled', 'in_progress', 'pickup_completed', 'in_transit', 'active', 'pending'] }
    });
    
    // Get completed transports today
    const startOfDay = new Date();
    startOfDay.setHours(0, 0, 0, 0);
    const completedToday = await Transport.countDocuments({
      $or: [
        { assignedTechnician: userId },
        { backupTechnician: userId }
      ],
      status: 'completed',
      completedAt: { $gte: startOfDay }
    });
    
    // Get completed transports this week
    const startOfWeek = new Date();
    startOfWeek.setDate(startOfWeek.getDate() - startOfWeek.getDay());
    startOfWeek.setHours(0, 0, 0, 0);
    const completedThisWeek = await Transport.countDocuments({
      $or: [
        { assignedTechnician: userId },
        { backupTechnician: userId }
      ],
      status: 'completed',
      completedAt: { $gte: startOfWeek }
    });
    
    // Calculate average completion time
    const completedTransports = await Transport.find({
      $or: [
        { assignedTechnician: userId },
        { backupTechnician: userId }
      ],
      status: 'completed',
      completedAt: { $exists: true },
      createdAt: { $exists: true }
    });
    
    const averageCompletionTime = completedTransports.length > 0
      ? completedTransports.reduce((sum, transport) => {
          const duration = new Date(transport.completedAt) - new Date(transport.createdAt);
          return sum + duration;
        }, 0) / completedTransports.length / (1000 * 60) // Convert to minutes
      : 0;
    
    res.json({
      success: true,
      stats: {
        totalTransports,
        activeTransports,
        completedToday,
        completedThisWeek,
        averageCompletionTime: Math.round(averageCompletionTime)
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to fetch technician stats',
      error: error.message
    });
  }
});

// Get transports for family member
router.get('/family-transports', authorize('family'), async (req, res) => {
  try {
    const transports = await Transport.find({
      'familyMembers.user': req.user._id
    })
    .populate('assignedTechnician', 'firstName lastName phone')
    .populate('deceased', 'firstName lastName uniqueIdTag')
    .sort({ createdAt: -1 })
    .exec();
    
    res.json({
      success: true,
      data: transports
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to fetch family transports',
      error: error.message
    });
  }
});

// Create new transport
router.post('/', authorize('funeral_home_staff', 'admin'), checkFeatureAccess('transportCreation'), async (req, res) => {
  try {
    console.log('Creating transport - User:', req.user.role, 'Permissions:', req.user.permissions);
    const {
      transportType,
      deceased,
      pickupLocation,
      deliveryLocation,
      scheduledPickupTime,
      scheduledDeliveryTime,
      assignedTechnician,
      familyMembers,
      priority,
      estimatedDuration
    } = req.body;
    
    // Generate unique case ID
    const caseId = `OB-${Date.now()}-${Math.random().toString(36).substr(2, 9).toUpperCase()}`;
    
    // Generate unique ID tag for deceased
    const uniqueIdTag = `TAG-${uuidv4().substr(0, 8).toUpperCase()}`;
    
    const transport = new Transport({
      caseId,
      transportType,
      deceased: {
        ...deceased,
        uniqueIdTag
      },
      pickupLocation,
      deliveryLocation,
      scheduledPickupTime,
      scheduledDeliveryTime,
      assignedTechnician,
      familyMembers,
      priority,
      estimatedDuration,
      organizationId: req.user.organizationId,
      createdBy: req.user._id
    });
    
    await transport.save();
    
    // Auto-generate comprehensive QR code for the transport
    let generatedQRCode = null;
    try {
      if (transport._id) {
        generatedQRCode = await qrCodeService.autoGenerateQRCodeForTransport(
          transport._id,
          req.user.organizationId,
          req.user._id
        );
        
        if (generatedQRCode) {
          console.log('QR code auto-generated for transport:', transport.caseId, 'QR ID:', generatedQRCode.qrCodeId);
        } else {
          console.log('QR code generation returned null for transport:', transport.caseId);
        }
      } else {
        console.error('Transport ID not available for QR code generation');
      }
    } catch (qrError) {
      console.error('Failed to auto-generate QR code:', qrError);
      // Continue without QR code to avoid breaking transport creation
    }
    
    // Record initial custody event on blockchain
    if (process.env.ENABLE_BLOCKCHAIN === 'true') {
      try {
        await blockchainService.recordCustodyEvent(
          transport.caseId,
          'transport_created',
          JSON.stringify({
            transportType,
            uniqueIdTag,
            createdBy: req.user._id,
            organizationId: req.user.organizationId
          })
        );
      } catch (blockchainError) {
        console.error('Failed to record blockchain event:', blockchainError);
      }
    }
    
    // Generate AI task reminders
    if (process.env.ENABLE_AI_VOICE === 'true') {
      try {
        const reminders = await aiService.generateTaskReminders(transport);
        transport.aiAssistant.taskReminders = reminders.reminders || [];
        await transport.save();
      } catch (aiError) {
        console.error('Failed to generate AI reminders:', aiError);
      }
    }
    
    res.status(201).json({
      success: true,
      data: transport,
      qrCode: generatedQRCode ? generatedQRCode.qrCodeImage : null,
      qrCodeId: generatedQRCode ? generatedQRCode.qrCodeId : null
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to create transport',
      error: error.message
    });
  }
});

// Get transport by ID
router.get('/:id', async (req, res) => {
  try {
    const transport = await Transport.findById(req.params.id)
      .populate('assignedTechnician', 'firstName lastName phone')
      .populate('backupTechnician', 'firstName lastName phone')
      .populate('familyMembers.user', 'firstName lastName email phone')
      .populate('custodyEvents.performedBy', 'firstName lastName')
      .populate('documents.uploadedBy', 'firstName lastName')
      .populate('emergencyAlerts.reportedBy', 'firstName lastName')
      .populate('notes.author', 'firstName lastName')
      .exec();
    
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check access permissions
    if (req.user.role === 'family') {
      const hasAccess = transport.familyMembers.some(fm => 
        fm.user._id.toString() === req.user._id.toString()
      );
      if (!hasAccess) {
        return res.status(403).json({
          success: false,
          message: 'Access denied'
        });
      }
    } else if (req.user.role !== 'admin' && 
               transport.organizationId.toString() !== req.user.organizationId?.toString()) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }
    
    res.json({
      success: true,
      data: transport
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to fetch transport',
      error: error.message
    });
  }
});

// Update transport status
router.patch('/:id/status', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { status, notes } = req.body;
    
    const transport = await Transport.findById(req.params.id);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check access permissions
    if (req.user.role !== 'admin' && 
        transport.organizationId.toString() !== req.user.organizationId?.toString()) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }
    
    const oldStatus = transport.status;
    transport.status = status;
    
    // Add status change note
    if (notes) {
      transport.notes.push({
        content: `Status changed from ${oldStatus} to ${status}: ${notes}`,
        author: req.user._id
      });
    }
    
    // Record custody event
    const custodyEvent = {
      eventType: 'status_updated',
      timestamp: new Date(),
      location: {
        coordinates: req.body.coordinates || null,
        address: req.body.address || null
      },
      performedBy: req.user._id,
      notes: `Status changed from ${oldStatus} to ${status}`
    };
    
    transport.custodyEvents.push(custodyEvent);
    
    // Update timestamps based on status
    if (status === 'pickup_completed' && !transport.actualPickupTime) {
      transport.actualPickupTime = new Date();
    }
    if (status === 'delivery_completed' && !transport.actualDeliveryTime) {
      transport.actualDeliveryTime = new Date();
    }
    if (status === 'completed' && !transport.completedAt) {
      transport.completedAt = new Date();
    }
    
    await transport.save();
    
    // Record blockchain event
    if (process.env.ENABLE_BLOCKCHAIN === 'true') {
      try {
        await blockchainService.recordCustodyEvent(
          transport.caseId,
          'status_updated',
          JSON.stringify({
            oldStatus,
            newStatus: status,
            performedBy: req.user._id,
            timestamp: new Date().toISOString()
          })
        );
      } catch (blockchainError) {
        console.error('Failed to record blockchain event:', blockchainError);
      }
    }
    
    res.json({
      success: true,
      data: transport
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to update transport status',
      error: error.message
    });
  }
});

// Add custody event
router.post('/:id/custody-events', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const {
      eventType,
      location,
      notes,
      photos,
      voiceLog,
      qrCodeScanned
    } = req.body;
    
    const transport = await Transport.findById(req.params.id);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check access permissions
    if (req.user.role !== 'admin' && 
        transport.organizationId.toString() !== req.user.organizationId?.toString()) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }
    
    const custodyEvent = {
      eventType,
      timestamp: new Date(),
      location,
      performedBy: req.user._id,
      notes,
      photos: photos || [],
      voiceLog: voiceLog || null,
      qrCodeScanned: qrCodeScanned || null
    };
    
    transport.custodyEvents.push(custodyEvent);
    await transport.save();
    
    // Record blockchain event
    if (process.env.ENABLE_BLOCKCHAIN === 'true') {
      try {
        await blockchainService.recordCustodyEvent(
          transport.caseId,
          eventType,
          JSON.stringify({
            location,
            notes,
            performedBy: req.user._id,
            timestamp: new Date().toISOString()
          })
        );
      } catch (blockchainError) {
        console.error('Failed to record blockchain event:', blockchainError);
      }
    }
    
    res.json({
      success: true,
      data: custodyEvent
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to add custody event',
      error: error.message
    });
  }
});

// Add GPS tracking point
router.post('/:id/gps-tracking', authorize('technician'), async (req, res) => {
  try {
    const { coordinates, speed, heading, accuracy } = req.body;
    
    const transport = await Transport.findById(req.params.id);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check if user is assigned to this transport
    if (transport.assignedTechnician.toString() !== req.user._id.toString() &&
        transport.backupTechnician?.toString() !== req.user._id.toString()) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }
    
    const gpsPoint = {
      timestamp: new Date(),
      coordinates,
      speed,
      heading,
      accuracy
    };
    
    transport.gpsTracking.push(gpsPoint);
    await transport.save();
    
    res.json({
      success: true,
      data: gpsPoint
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to add GPS tracking point',
      error: error.message
    });
  }
});

// Add emergency alert
router.post('/:id/emergency-alerts', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { type, description } = req.body;
    
    const transport = await Transport.findById(req.params.id);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    const emergencyAlert = {
      type,
      description,
      timestamp: new Date(),
      reportedBy: req.user._id
    };
    
    transport.emergencyAlerts.push(emergencyAlert);
    transport.status = 'emergency';
    await transport.save();
    
    // TODO: Send notifications to relevant parties
    
    res.json({
      success: true,
      data: emergencyAlert
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to add emergency alert',
      error: error.message
    });
  }
});

// Get transport QR code
router.get('/:id/qr-code', async (req, res) => {
  try {
    const transport = await Transport.findById(req.params.id);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    const qrCodeData = {
      caseId: transport.caseId,
      uniqueIdTag: transport.deceased.uniqueIdTag,
      type: 'transport'
    };
    
    const qrCodeUrl = await QRCode.toDataURL(JSON.stringify(qrCodeData), {
      width: 256,
      margin: 2
    });
    
    res.json({
      success: true,
      data: {
        qrCode: qrCodeUrl,
        caseId: transport.caseId,
        uniqueIdTag: transport.deceased.uniqueIdTag
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to generate QR code',
      error: error.message
    });
  }
});

// Get blockchain custody events
router.get('/:id/blockchain-events', async (req, res) => {
  try {
    const transport = await Transport.findById(req.params.id);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    if (process.env.ENABLE_BLOCKCHAIN !== 'true') {
      return res.status(400).json({
        success: false,
        message: 'Blockchain is not enabled'
      });
    }
    
    const blockchainEvents = await blockchainService.getCustodyEvents(transport.caseId);
    
    res.json({
      success: true,
      data: blockchainEvents
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to fetch blockchain events',
      error: error.message
    });
  }
});

// Verify custody chain
router.get('/:id/verify-custody', async (req, res) => {
  try {
    const transport = await Transport.findById(req.params.id);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    if (process.env.ENABLE_BLOCKCHAIN !== 'true') {
      return res.status(400).json({
        success: false,
        message: 'Blockchain is not enabled'
      });
    }
    
    const verification = await blockchainService.verifyCustodyChain(transport.caseId);
    
    res.json({
      success: true,
      data: verification
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to verify custody chain',
      error: error.message
    });
  }
});

// Update transport status
router.put('/:id/status', authorize('technician', 'funeral_home_staff', 'admin'), async (req, res) => {
  try {
    const { status, startedAt, completedAt, notes } = req.body;
    const transportId = req.params.id;
    
    // Find the transport
    const transport = await Transport.findById(transportId);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check if user is authorized to update this transport
    const isAuthorized = 
      req.user.role === 'admin' || 
      req.user.role === 'funeral_home_staff' ||
      transport.assignedTechnician.toString() === req.user._id.toString() ||
      transport.backupTechnician?.toString() === req.user._id.toString();
    
    if (!isAuthorized) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to update this transport'
      });
    }
    
    // Update status and related fields
    const updateData = { status };
    
    if (status === 'in_progress' && startedAt) {
      updateData.actualPickupTime = new Date(startedAt);
    }
    
    if (status === 'completed' && completedAt) {
      updateData.actualDeliveryTime = new Date(completedAt);
      updateData.completedAt = new Date(completedAt);
    }
    
    if (notes) {
      updateData.notes = notes;
    }
    
    // Add custody event
    const custodyEvent = {
      eventType: status === 'in_progress' ? 'pickup_initiated' : 
                 status === 'pickup_completed' ? 'pickup_completed' :
                 status === 'in_transit' ? 'transport_started' :
                 status === 'delivery_completed' ? 'delivery_arrived' :
                 status === 'completed' ? 'delivery_completed' : 'status_updated',
      timestamp: new Date(),
      location: {
        latitude: req.body.latitude,
        longitude: req.body.longitude,
        address: req.body.address
      },
      performedBy: req.user._id,
      notes: notes || `Status updated to ${status}`
    };
    
    updateData.$push = { custodyEvents: custodyEvent };
    
    const updatedTransport = await Transport.findByIdAndUpdate(
      transportId,
      updateData,
      { new: true }
    ).populate('assignedTechnician', 'firstName lastName');
    
    // Record blockchain event if enabled
    if (process.env.ENABLE_BLOCKCHAIN === 'true') {
      try {
        await blockchainService.recordCustodyEvent(
          transport.caseId,
          custodyEvent.eventType,
          JSON.stringify({
            status,
            performedBy: req.user._id,
            timestamp: custodyEvent.timestamp,
            location: custodyEvent.location,
            notes: custodyEvent.notes
          })
        );
      } catch (blockchainError) {
        console.error('Failed to record blockchain event:', blockchainError);
      }
    }
    
    res.json({
      success: true,
      message: 'Transport status updated successfully',
      data: updatedTransport
    });
  } catch (error) {
    console.error('Error updating transport status:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update transport status',
      error: error.message
    });
  }
});

// GPS tracking update
router.post('/:id/gps-track', authorize('technician'), async (req, res) => {
  try {
    const { latitude, longitude, address, timestamp } = req.body;
    const transportId = req.params.id;
    
    // Find the transport
    const transport = await Transport.findById(transportId);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check if user is authorized to track this transport
    const isAuthorized = 
      transport.assignedTechnician.toString() === req.user._id.toString() ||
      transport.backupTechnician?.toString() === req.user._id.toString();
    
    if (!isAuthorized) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to track this transport'
      });
    }
    
    // Add GPS tracking point
    const gpsPoint = {
      timestamp: timestamp ? new Date(timestamp) : new Date(),
      coordinates: {
        latitude: parseFloat(latitude),
        longitude: parseFloat(longitude)
      },
      address: address
    };
    
    // Add to GPS tracking array
    transport.gpsTracking.push(gpsPoint);
    
    // Add custody event for GPS checkpoint
    const custodyEvent = {
      eventType: 'gps_checkpoint',
      timestamp: gpsPoint.timestamp,
      location: {
        coordinates: gpsPoint.coordinates,
        address: gpsPoint.address
      },
      performedBy: req.user._id,
      notes: `GPS checkpoint recorded at ${gpsPoint.address || 'unknown location'}`
    };
    
    transport.custodyEvents.push(custodyEvent);
    
    await transport.save();
    
    // Record blockchain event if enabled
    if (process.env.ENABLE_BLOCKCHAIN === 'true') {
      try {
        await blockchainService.recordCustodyEvent(
          transport.caseId,
          'gps_checkpoint',
          JSON.stringify({
            latitude: gpsPoint.coordinates.latitude,
            longitude: gpsPoint.coordinates.longitude,
            address: gpsPoint.address,
            timestamp: gpsPoint.timestamp,
            performedBy: req.user._id
          })
        );
      } catch (blockchainError) {
        console.error('Failed to record blockchain GPS event:', blockchainError);
      }
    }
    
    // Send real-time notification to family and admin
    try {
      // Notify family members
      if (transport.familyMembers && transport.familyMembers.length > 0) {
        // This would integrate with your notification service
        console.log('Sending GPS update to family members');
      }
      
      // Notify admin dashboard
      // This would integrate with your WebSocket service
      console.log('Sending GPS update to admin dashboard');
    } catch (notificationError) {
      console.error('Failed to send notifications:', notificationError);
    }
    
    res.json({
      success: true,
      message: 'GPS tracking point recorded',
      data: gpsPoint
    });
  } catch (error) {
    console.error('Error recording GPS tracking:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to record GPS tracking',
      error: error.message
    });
  }
});

// Photo capture and upload
router.post('/:id/photo', authorize('technician'), async (req, res) => {
  try {
    const { photoData, eventType, notes, latitude, longitude, address } = req.body;
    const transportId = req.params.id;
    
    // Find the transport
    const transport = await Transport.findById(transportId);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check authorization
    const isAuthorized = 
      transport.assignedTechnician.toString() === req.user._id.toString() ||
      transport.backupTechnician?.toString() === req.user._id.toString();
    
    if (!isAuthorized) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to add photos to this transport'
      });
    }
    
    // Upload photo to IPFS
    let ipfsHash = null;
    if (process.env.ENABLE_IPFS === 'true') {
      try {
        const ipfsService = require('../services/ipfs');
        ipfsHash = await ipfsService.uploadPhoto(photoData);
      } catch (ipfsError) {
        console.error('Failed to upload to IPFS:', ipfsError);
      }
    }
    
    // Create photo metadata
    const photoMetadata = {
      url: req.body.photoUrl || null,
      ipfsHash: ipfsHash,
      metadata: {
        timestamp: new Date(),
        gps: {
          latitude: parseFloat(latitude),
          longitude: parseFloat(longitude)
        },
        eventType: eventType,
        notes: notes
      }
    };
    
    // Add custody event with photo
    const custodyEvent = {
      eventType: eventType || 'photo_documentation',
      timestamp: new Date(),
      location: {
        coordinates: {
          latitude: parseFloat(latitude),
          longitude: parseFloat(longitude)
        },
        address: address
      },
      performedBy: req.user._id,
      notes: notes || `Photo documentation for ${eventType}`,
      photos: [photoMetadata]
    };
    
    transport.custodyEvents.push(custodyEvent);
    await transport.save();
    
    // Record blockchain event
    if (process.env.ENABLE_BLOCKCHAIN === 'true') {
      try {
        await blockchainService.recordCustodyEvent(
          transport.caseId,
          'photo_documentation',
          JSON.stringify({
            eventType: eventType,
            ipfsHash: ipfsHash,
            latitude: parseFloat(latitude),
            longitude: parseFloat(longitude),
            timestamp: new Date(),
            performedBy: req.user._id
          })
        );
      } catch (blockchainError) {
        console.error('Failed to record blockchain photo event:', blockchainError);
      }
    }
    
    res.json({
      success: true,
      message: 'Photo captured and recorded',
      data: {
        photoMetadata,
        custodyEvent
      }
    });
  } catch (error) {
    console.error('Error capturing photo:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to capture photo',
      error: error.message
    });
  }
});

// Voice logging
router.post('/:id/voice-log', authorize('technician'), async (req, res) => {
  try {
    const { voiceData, eventType, notes, duration } = req.body;
    const transportId = req.params.id;
    
    // Find the transport
    const transport = await Transport.findById(transportId);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check authorization
    const isAuthorized = 
      transport.assignedTechnician.toString() === req.user._id.toString() ||
      transport.backupTechnician?.toString() === req.user._id.toString();
    
    if (!isAuthorized) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to add voice logs to this transport'
      });
    }
    
    // Upload voice to IPFS
    let ipfsHash = null;
    if (process.env.ENABLE_IPFS === 'true') {
      try {
        const ipfsService = require('../services/ipfs');
        ipfsHash = await ipfsService.uploadVoice(voiceData);
      } catch (ipfsError) {
        console.error('Failed to upload voice to IPFS:', ipfsError);
      }
    }
    
    // Create voice log metadata
    const voiceLog = {
      url: req.body.voiceUrl || null,
      ipfsHash: ipfsHash,
      duration: duration || 0
    };
    
    // Add custody event with voice log
    const custodyEvent = {
      eventType: eventType || 'voice_log',
      timestamp: new Date(),
      performedBy: req.user._id,
      notes: notes || 'Voice log recorded',
      voiceLog: voiceLog
    };
    
    transport.custodyEvents.push(custodyEvent);
    await transport.save();
    
    // Record blockchain event
    if (process.env.ENABLE_BLOCKCHAIN === 'true') {
      try {
        await blockchainService.recordCustodyEvent(
          transport.caseId,
          'voice_log',
          JSON.stringify({
            eventType: eventType,
            ipfsHash: ipfsHash,
            duration: duration,
            timestamp: new Date(),
            performedBy: req.user._id
          })
        );
      } catch (blockchainError) {
        console.error('Failed to record blockchain voice event:', blockchainError);
      }
    }
    
    res.json({
      success: true,
      message: 'Voice log recorded',
      data: {
        voiceLog,
        custodyEvent
      }
    });
  } catch (error) {
    console.error('Error recording voice log:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to record voice log',
      error: error.message
    });
  }
});

// QR code scanning
router.post('/:id/scan-qr', authorize('technician'), async (req, res) => {
  try {
    const { qrCode, location, notes } = req.body;
    const transportId = req.params.id;
    
    // Find the transport
    const transport = await Transport.findById(transportId);
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check authorization
    const isAuthorized = 
      transport.assignedTechnician.toString() === req.user._id.toString() ||
      transport.backupTechnician?.toString() === req.user._id.toString();
    
    if (!isAuthorized) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to scan QR codes for this transport'
      });
    }
    
    // Verify QR code matches transport
    const expectedQRData = {
      caseId: transport.caseId,
      uniqueIdTag: transport.deceased.uniqueIdTag,
      type: 'transport'
    };
    
    let scannedData;
    try {
      scannedData = JSON.parse(qrCode);
    } catch (parseError) {
      return res.status(400).json({
        success: false,
        message: 'Invalid QR code format'
      });
    }
    
    if (scannedData.caseId !== transport.caseId || 
        scannedData.uniqueIdTag !== transport.deceased.uniqueIdTag) {
      return res.status(400).json({
        success: false,
        message: 'QR code does not match this transport'
      });
    }
    
    // Add custody event with QR scan
    const custodyEvent = {
      eventType: 'qr_code_scanned',
      timestamp: new Date(),
      performedBy: req.user._id,
      notes: notes || `QR code scanned at ${location}`,
      qrCodeScanned: {
        code: qrCode,
        location: location
      }
    };
    
    transport.custodyEvents.push(custodyEvent);
    await transport.save();
    
    // Record blockchain event
    if (process.env.ENABLE_BLOCKCHAIN === 'true') {
      try {
        await blockchainService.recordCustodyEvent(
          transport.caseId,
          'qr_code_scanned',
          JSON.stringify({
            qrCode: qrCode,
            location: location,
            timestamp: new Date(),
            performedBy: req.user._id
          })
        );
      } catch (blockchainError) {
        console.error('Failed to record blockchain QR event:', blockchainError);
      }
    }
    
    res.json({
      success: true,
      message: 'QR code scanned and verified',
      data: {
        scannedData,
        custodyEvent
      }
    });
  } catch (error) {
    console.error('Error scanning QR code:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to scan QR code',
      error: error.message
    });
  }
});

// Emergency alert
router.post('/:id/emergency', authorize('technician'), async (req, res) => {
  try {
    const { emergencyType, message, latitude, longitude, address } = req.body;
    const transportId = req.params.id;
    
    // Find the transport
    const transport = await Transport.findById(transportId)
      .populate('assignedTechnician', 'firstName lastName phone')
      .populate('organizationId', 'name contactPhone');
    
    if (!transport) {
      return res.status(404).json({
        success: false,
        message: 'Transport not found'
      });
    }
    
    // Check authorization
    const isAuthorized = 
      transport.assignedTechnician._id.toString() === req.user._id.toString() ||
      transport.backupTechnician?.toString() === req.user._id.toString();
    
    if (!isAuthorized) {
      return res.status(403).json({
        success: false,
        message: 'Not authorized to send emergency alerts for this transport'
      });
    }
    
    // Update transport status to emergency
    transport.status = 'emergency';
    
    // Add custody event for emergency
    const custodyEvent = {
      eventType: 'emergency_stop',
      timestamp: new Date(),
      location: {
        coordinates: {
          latitude: parseFloat(latitude),
          longitude: parseFloat(longitude)
        },
        address: address
      },
      performedBy: req.user._id,
      notes: `Emergency alert: ${emergencyType} - ${message}`
    };
    
    transport.custodyEvents.push(custodyEvent);
    await transport.save();
    
    // Record blockchain event
    if (process.env.ENABLE_BLOCKCHAIN === 'true') {
      try {
        await blockchainService.recordCustodyEvent(
          transport.caseId,
          'emergency_stop',
          JSON.stringify({
            emergencyType: emergencyType,
            message: message,
            latitude: parseFloat(latitude),
            longitude: parseFloat(longitude),
            timestamp: new Date(),
            performedBy: req.user._id
          })
        );
      } catch (blockchainError) {
        console.error('Failed to record blockchain emergency event:', blockchainError);
      }
    }
    
    // Send emergency notifications
    try {
      // Notify admin immediately
      console.log(`EMERGENCY ALERT: Transport ${transport.caseId} - ${emergencyType}`);
      console.log(`Location: ${address} (${latitude}, ${longitude})`);
      console.log(`Technician: ${transport.assignedTechnician.firstName} ${transport.assignedTechnician.lastName}`);
      console.log(`Message: ${message}`);
      
      // Notify family members
      if (transport.familyMembers && transport.familyMembers.length > 0) {
        console.log('Sending emergency notification to family members');
      }
      
      // Notify organization
      if (transport.organizationId) {
        console.log(`Notifying organization: ${transport.organizationId.name}`);
      }
    } catch (notificationError) {
      console.error('Failed to send emergency notifications:', notificationError);
    }
    
    res.json({
      success: true,
      message: 'Emergency alert sent successfully',
      data: {
        emergencyType,
        message,
        location: { latitude, longitude, address },
        custodyEvent
      }
    });
  } catch (error) {
    console.error('Error sending emergency alert:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to send emergency alert',
      error: error.message
    });
  }
});

module.exports = router;
