const socketIo = require('socket.io');
const jwt = require('jsonwebtoken');
const User = require('../models/User');

let io;

// Initialize Socket.IO
const initializeSocket = (server) => {
  io = socketIo(server, {
    cors: {
      origin: process.env.ALLOWED_ORIGINS?.split(',') || ['http://localhost:3000', 'http://localhost:19006'],
      methods: ['GET', 'POST'],
      credentials: true
    },
    transports: ['websocket', 'polling']
  });

  // Authentication middleware
  io.use(async (socket, next) => {
    try {
      const token = socket.handshake.auth.token || socket.handshake.headers.authorization?.split(' ')[1];
      
      if (!token) {
        return next(new Error('Authentication error: No token provided'));
      }

      const decoded = jwt.verify(token, process.env.JWT_SECRET);
      const user = await User.findById(decoded.id).select('-password');
      
      if (!user || !user.isActive) {
        return next(new Error('Authentication error: Invalid user'));
      }

      socket.user = user;
      next();
    } catch (error) {
      next(new Error('Authentication error: Invalid token'));
    }
  });

  // Connection handler
  io.on('connection', (socket) => {
    console.log(`User connected: ${socket.user.email} (${socket.user.role})`);

    // Join user to appropriate rooms based on role
    joinUserToRooms(socket);

    // Handle transport updates
    socket.on('join-transport', (transportId) => {
      socket.join(`transport-${transportId}`);
      console.log(`User ${socket.user.email} joined transport ${transportId}`);
    });

    socket.on('leave-transport', (transportId) => {
      socket.leave(`transport-${transportId}`);
      console.log(`User ${socket.user.email} left transport ${transportId}`);
    });

    // Handle organization updates
    socket.on('join-organization', (organizationId) => {
      socket.join(`organization-${organizationId}`);
      console.log(`User ${socket.user.email} joined organization ${organizationId}`);
    });

    // Handle family member updates
    socket.on('join-family', (transportId) => {
      socket.join(`family-${transportId}`);
      console.log(`User ${socket.user.email} joined family room for transport ${transportId}`);
    });

    // Handle real-time location updates
    socket.on('location-update', (data) => {
      const { transportId, location } = data;
      socket.to(`transport-${transportId}`).emit('location-updated', {
        transportId,
        location,
        timestamp: new Date().toISOString(),
        userId: socket.user._id
      });
    });

    // Handle emergency alerts
    socket.on('emergency-alert', (data) => {
      const { transportId, alert } = data;
      socket.to(`transport-${transportId}`).emit('emergency-alert-received', {
        transportId,
        alert,
        timestamp: new Date().toISOString(),
        userId: socket.user._id
      });
    });

    // Handle custody event updates
    socket.on('custody-event', (data) => {
      const { transportId, event } = data;
      socket.to(`transport-${transportId}`).emit('custody-event-updated', {
        transportId,
        event,
        timestamp: new Date().toISOString(),
        userId: socket.user._id
      });
    });

    // Handle chat messages
    socket.on('chat-message', (data) => {
      const { transportId, message } = data;
      socket.to(`transport-${transportId}`).emit('chat-message-received', {
        transportId,
        message: {
          ...message,
          sender: {
            id: socket.user._id,
            name: `${socket.user.firstName} ${socket.user.lastName}`,
            role: socket.user.role
          },
          timestamp: new Date().toISOString()
        }
      });
    });

    // Handle voice call requests
    socket.on('voice-call-request', (data) => {
      const { transportId, recipientId } = data;
      socket.to(`user-${recipientId}`).emit('voice-call-incoming', {
        transportId,
        caller: {
          id: socket.user._id,
          name: `${socket.user.firstName} ${socket.user.lastName}`,
          role: socket.user.role
        },
        timestamp: new Date().toISOString()
      });
    });

    // Handle voice call responses
    socket.on('voice-call-response', (data) => {
      const { transportId, callerId, accepted } = data;
      socket.to(`user-${callerId}`).emit('voice-call-response-received', {
        transportId,
        accepted,
        responder: {
          id: socket.user._id,
          name: `${socket.user.firstName} ${socket.user.lastName}`,
          role: socket.user.role
        },
        timestamp: new Date().toISOString()
      });
    });

    // Handle AI voice assistant requests
    socket.on('ai-voice-request', (data) => {
      const { transportId, audioData, context } = data;
      // Process AI voice request and emit response
      socket.emit('ai-voice-response', {
        transportId,
        response: 'AI voice response placeholder',
        timestamp: new Date().toISOString()
      });
    });

    // Handle disconnect
    socket.on('disconnect', () => {
      console.log(`User disconnected: ${socket.user.email}`);
    });
  });

  return io;
};

// Join user to appropriate rooms based on role
const joinUserToRooms = (socket) => {
  const user = socket.user;

  // Join user's personal room
  socket.join(`user-${user._id}`);

  // Join organization room if user belongs to one
  if (user.organizationId) {
    socket.join(`organization-${user.organizationId}`);
  }

  // Join role-specific rooms
  switch (user.role) {
    case 'admin':
      socket.join('admin-room');
      break;
    case 'funeral_home_staff':
      socket.join('staff-room');
      if (user.organizationId) {
        socket.join(`staff-org-${user.organizationId}`);
      }
      break;
    case 'technician':
      socket.join('technician-room');
      if (user.organizationId) {
        socket.join(`technician-org-${user.organizationId}`);
      }
      break;
    case 'family':
      socket.join('family-room');
      break;
  }
};

// Emit functions for server-side events
const emitTransportUpdate = (transportId, update) => {
  if (io) {
    io.to(`transport-${transportId}`).emit('transport-updated', {
      transportId,
      update,
      timestamp: new Date().toISOString()
    });
  }
};

const emitStatusChange = (transportId, status, userId) => {
  if (io) {
    io.to(`transport-${transportId}`).emit('status-changed', {
      transportId,
      status,
      userId,
      timestamp: new Date().toISOString()
    });
  }
};

const emitCustodyEvent = (transportId, event) => {
  if (io) {
    io.to(`transport-${transportId}`).emit('custody-event-recorded', {
      transportId,
      event,
      timestamp: new Date().toISOString()
    });
  }
};

const emitEmergencyAlert = (transportId, alert) => {
  if (io) {
    io.to(`transport-${transportId}`).emit('emergency-alert', {
      transportId,
      alert,
      timestamp: new Date().toISOString()
    });
  }
};

const emitNotification = (userId, notification) => {
  if (io) {
    io.to(`user-${userId}`).emit('notification', {
      notification,
      timestamp: new Date().toISOString()
    });
  }
};

const emitFamilyUpdate = (transportId, update) => {
  if (io) {
    io.to(`family-${transportId}`).emit('family-update', {
      transportId,
      update,
      timestamp: new Date().toISOString()
    });
  }
};

const emitOrganizationUpdate = (organizationId, update) => {
  if (io) {
    io.to(`organization-${organizationId}`).emit('organization-updated', {
      organizationId,
      update,
      timestamp: new Date().toISOString()
    });
  }
};

const emitBlockchainEvent = (transportId, event) => {
  if (io) {
    io.to(`transport-${transportId}`).emit('blockchain-event', {
      transportId,
      event,
      timestamp: new Date().toISOString()
    });
  }
};

// Get connected users count
const getConnectedUsersCount = () => {
  if (io) {
    return io.engine.clientsCount;
  }
  return 0;
};

// Get connected users by role
const getConnectedUsersByRole = (role) => {
  if (io) {
    const sockets = io.sockets.sockets;
    const users = [];
    
    sockets.forEach((socket) => {
      if (socket.user && socket.user.role === role) {
        users.push({
          id: socket.user._id,
          email: socket.user.email,
          name: `${socket.user.firstName} ${socket.user.lastName}`,
          role: socket.user.role,
          organizationId: socket.user.organizationId
        });
      }
    });
    
    return users;
  }
  return [];
};

module.exports = {
  initializeSocket,
  emitTransportUpdate,
  emitStatusChange,
  emitCustodyEvent,
  emitEmergencyAlert,
  emitNotification,
  emitFamilyUpdate,
  emitOrganizationUpdate,
  emitBlockchainEvent,
  getConnectedUsersCount,
  getConnectedUsersByRole
};
