const fs = require('fs');
const PinataSDK = require('@pinata/sdk');
require('dotenv').config({ path: './.env' });

async function testPinataSDK() {
  console.log('=== Testing Pinata SDK ===\n');
  
  const pinataConfig = {
    apiKey: process.env.PINATA_API_KEY,
    secretApiKey: process.env.PINATA_SECRET_KEY || process.env.PINATA_SECRET_API_KEY,
    jwtToken: process.env.PINATA_JWT_TOKEN || process.env.PINATA_JWT_SECRET,
  };
  
  console.log('Pinata Config:', {
    apiKey: pinataConfig.apiKey ? `${pinataConfig.apiKey.substring(0, 10)}...` : 'NOT SET',
    secretApiKey: pinataConfig.secretApiKey ? `${pinataConfig.secretApiKey.substring(0, 10)}...` : 'NOT SET',
    jwtToken: pinataConfig.jwtToken ? 'SET' : 'NOT SET'
  });
  
  try {
    // Create a simple test file
    const testContent = 'This is a test file for Pinata upload';
    const testFileName = 'test-file.txt';
    const testFilePath = './test-file.txt';
    
    fs.writeFileSync(testFilePath, testContent);
    console.log('✅ Created test file:', testFilePath);
    
    // Test 1: Using API Key + Secret
    if (pinataConfig.apiKey && pinataConfig.secretApiKey) {
      console.log('\n🔐 Test 1: Using API Key + Secret...');
      const pinata = new PinataSDK(pinataConfig.apiKey, pinataConfig.secretApiKey);
      
      try {
        const result = await pinata.pinFileToIPFS(fs.createReadStream(testFilePath), {
          pinataMetadata: {
            name: testFileName,
            keyvalues: {
              service: 'obitix',
              timestamp: new Date().toISOString()
            }
          }
        });
        
        console.log('✅ Success with API Key + Secret:', result);
      } catch (error) {
        console.log('❌ Error with API Key + Secret:', error.message);
      }
    }
    
    // Test 2: Using JWT Token
    if (pinataConfig.jwtToken) {
      console.log('\n🔐 Test 2: Using JWT Token...');
      const pinata = new PinataSDK({ pinataJWTKey: pinataConfig.jwtToken });
      
      try {
        const result = await pinata.pinFileToIPFS(fs.createReadStream(testFilePath), {
          pinataMetadata: {
            name: testFileName,
            keyvalues: {
              service: 'obitix',
              timestamp: new Date().toISOString()
            }
          }
        });
        
        console.log('✅ Success with JWT Token:', result);
      } catch (error) {
        console.log('❌ Error with JWT Token:', error.message);
      }
    }
    
    // Clean up test file
    fs.unlinkSync(testFilePath);
    console.log('✅ Cleaned up test file');
    
  } catch (error) {
    console.error('❌ Test failed:', error.message);
    
    // Clean up test file if it exists
    if (fs.existsSync('./test-file.txt')) {
      fs.unlinkSync('./test-file.txt');
    }
  }
}

testPinataSDK();
